// license:BSD-3-Clause
// copyright-holders:m1macrophage

/*
The Prophet 5 (aka Model 1000) is a digitally-controlled, 5-voice analog
synthesizer.

The firwmare, running on a Z80, is responsible for:
* Scanning the keyboard and buttons.
* Scanning the value of potentiometers.
* Driving LEDs and digit displays.
* Reacting to inputs (footswitch, external CV and gate) and driving outputs
  (CV, gate).
* Cassette I/O.
* Voice control: setting control voltages (CVs) for voice parameters, asserting
  voice gates, and routing signals via CMOS switches.

The rest of the description is for the Prophet 5 Rev 3.0, which is the revision
being emulated. Most of this info should also apply to Revs 3.1, 3.2 and 3.3.

CVs are generated by DAC71-CSB-I, a 16-bit unipolar current output DAC. Some
units used the DAC71-CSB-V, which outputs a voltage instead. Only the 14 MSbits
of the DAC are utilized. Frequency CVs use the full 14 bits, whereas other CVs
just use the 7 MSbits. See update_vdac() for details.

CVs are routed to 38 sample & hold (S&H) circuits via CD4051 MUXes. There are 3
CVs for each of the 5 voices (3 x 5 = 15 CVs), controlling oscillator and filter
frequencies. The rest (23 CVs) are common to all voices. This setup results in a
1-part multitimbrality. See update_sh() for details.

There is no dedicated ADC chip. The knobs and external CV are scanned by routing
the knob (or input) voltages to a window comparator. Those voltages are compared
with the ADC reference, which is controlled by the firmware. See update_vmux()
and adc_r() for details.

TODO: Outline of voice architecture.

This driver is based on the Prophet 5 Rev 3.0 technical manual, and is intended
as an education tool. To that end, names for variables, I/O handlers, enums,
etc., generally match signal names in the schematics.

There is minimal audio. Running with `-oslog -output console` will display CVs,
and voice and LED control signals.

When the Prophet 5 boots up, it runs its autotune routine (`tune` LED will be
illuminated). The synth is unresponsive while this is happening. Once that's
done, it will load bank 1 program 1. To instantly configure the sound based on
the current knob positions, press the "preset" button ('P') to exit preset mode
(preset LED should turn off).
*/

#include "emu.h"

#include "cpu/z80/z80.h"
#include "machine/7474.h"
#include "machine/pit8253.h"
#include "machine/nvram.h"
#include "machine/output_latch.h"
#include "machine/rescap.h"
#include "machine/timer.h"
#include "sound/cem3310.h"
#include "sound/cem3320.h"
#include "sound/dac.h"
#include "sound/flt_rc.h"
#include "sound/mixer.h"
#include "sound/mm5837.h"
#include "sound/va_ops.h"
#include "sound/va_vca.h"
#include "video/pwm.h"

#include "speaker.h"

#include "corestr.h"

#include "sequential_prophet5.lh"

#define LOG_SWITCHES    (1U << 1)
#define LOG_CV          (1U << 2)
#define LOG_ADC         (1U << 3)
#define LOG_GATE        (1U << 4)
#define LOG_CALIBRATION (1U << 5)
#define LOG_PROG_LATCH  (1U << 6)
#define LOG_FILTER      (1U << 7)
#define LOG_WHEEL       (1U << 8)

#define VERBOSE (LOG_GENERAL | LOG_CALIBRATION | LOG_PROG_LATCH | LOG_CV)
//#define LOG_OUTPUT_FUNC osd_printf_info

#include "logmacro.h"

namespace {

constexpr double VCC = 5.0;
constexpr double VPLUS = 15.0;
constexpr double VMINUS = -15.0;
constexpr double MAX_CV_IN = 10.0;  // Maximun voltage for CV inputs on back panel.

double normalized(const required_ioport &input)
{
	assert(input->field(1)->minval() == 0);
	return double(input->read()) / double(input->field(1)->maxval());
}

// A voltage-to-current converter based on a PNP transistor. Converts control
// voltages to control currents for CA3280 OTAs.
//
// CV---R---|-->PNP--|---Rout---Iabc pin of CA3280
//               |
//              GND
//
// As long as the value of Rout is small enough, it won't have a meaningful
// effect on the current. So it is ignored here.
double cv2cc(double cv, double r)
{
	// These values were chosen to work for the Rs found on the Prophet 5.
	constexpr double CV_ZERO = 0.3;
	constexpr double CV_LINEAR = 0.8;
	constexpr double QVBE = 0.6;

	if (cv <= CV_ZERO)
	{
		// Very small control voltage. Current is < 1nA. Treat as 0.
		return 0;
	}
	else if (cv >= CV_LINEAR)
	{
		// Large control voltage. Current is essentially linear wrt voltage.
		return (cv - QVBE) / r;
	}
	else
	{
		// Small control voltage. The relationship between current and voltage
		// is non-linear. Getting an accurate response would require iterative
		// approximations.

		// Instead, we use a quadratic bezier curve with endpoints P0 and P2,
		// and control point P1. This gives a smooth transition between the line
		// used when cv <= CV_ZERO, and the line used when cv >= CV_LINEAR.

		constexpr double P0[2] = { 0.3, 0.0 };
		constexpr double P1[2] = { 0.6, 0.0 };
		const double P2[2] = { 0.8, (0.8 - QVBE) / r };

		const double t = (cv - P0[0]) / (P2[0] - P0[0]);
		const double t1 = 1.0 - t;
		return t1 * t1 * P0[1] + 2.0 * t1 * t * P1[1] + t * t * P2[1];
	}
}

// A streaming version of the cv2cc() function above.
class prophet5_cv2cc_device :  public device_t, public device_sound_interface
{
public:
	prophet5_cv2cc_device(const machine_config &mconfig, const char *tag, device_t *owner, double r) ATTR_COLD;
	prophet5_cv2cc_device(const machine_config &mconfig, const char *tag, device_t *owner, u32 clock) ATTR_COLD;

protected:
	void device_start() override ATTR_COLD;
	void sound_stream_update(sound_stream &stream) override;

	const double m_r;
	sound_stream *m_stream;
};

}  // anonymous namespace

DEFINE_DEVICE_TYPE(PROPHET5_CV2CC, prophet5_cv2cc_device, "prophet5_cv2cc", "Prophet 5 voltage-to-current converter")

prophet5_cv2cc_device::prophet5_cv2cc_device(const machine_config &mconfig, const char *tag, device_t *owner, double r)
	: device_t(mconfig, PROPHET5_CV2CC, tag, owner, 0)
	, device_sound_interface(mconfig, *this)
	, m_r(r)
	, m_stream(nullptr)
{
}

prophet5_cv2cc_device::prophet5_cv2cc_device(const machine_config &mconfig, const char *tag, device_t *owner, u32 clock)
	: prophet5_cv2cc_device(mconfig, tag, owner, RES_K(4.7))
{
}

void prophet5_cv2cc_device::device_start()
{
	m_stream = stream_alloc(1, 1, machine().sample_rate());
}

void prophet5_cv2cc_device::sound_stream_update(sound_stream &stream)
{
	for (int i = 0; i < stream.samples(); ++i)
		stream.put(0, i, cv2cc(stream.get(0, i), m_r));
}


namespace {

// A Prophet 5 Rev 3.x voice.
class prophet5_voice_device : public device_t, public device_sound_interface
{
public:
	prophet5_voice_device(
		const machine_config &mconfig,
		const char *tag,
		device_t *owner,
		device_sound_interface *filt_sum_cv,
		device_sound_interface *noise) ATTR_COLD;
	prophet5_voice_device(const machine_config &mconfig, const char *tag, device_t *owner, u32 clock) ATTR_COLD;

	auto volume_changed_cb() { return m_volume_changed_cb.bind(); }

	cem3310_device *vca_eg() { return m_vca_eg.target(); }
	cem3310_device *filt_eg() { return m_filt_eg.target(); }

	double volume_trimmer_r() const;

	void gate_w(int state);
	void filt_sh_w(double cv);
	void filt_res_w(double cv);
	void filt_env_amt_w(double cc);

	DECLARE_INPUT_CHANGED_MEMBER(filter_trimmer_changed) { update_filter_freq_calibration(); }
	DECLARE_INPUT_CHANGED_MEMBER(volume_trimmer_changed) { m_volume_changed_cb(0); }

	const char *trimmer_name_volume() const ATTR_COLD { return m_volume_name.c_str(); }
	const char *trimmer_name_filt_scale() const ATTR_COLD { return m_filt_scale_name.c_str(); }
	const char *trimmer_name_filt_offset() const ATTR_COLD { return m_filt_offset_name.c_str(); }

protected:
	void device_add_mconfig(machine_config &config) override ATTR_COLD;
	ioport_constructor device_input_ports() const override ATTR_COLD;
	void device_start() override ATTR_COLD;
	void device_reset() override ATTR_COLD;

	void sound_stream_update(sound_stream &stream) override;

private:
	void update_filter_freq_calibration();

	const std::string m_volume_name;
	const std::string m_filt_scale_name;
	const std::string m_filt_offset_name;

	device_sound_interface *const m_filt_sum_cv;
	device_sound_interface *const m_noise;
	sound_stream *m_stream = nullptr;

	required_device<cem3310_device> m_vca_eg;  // U412
	required_device<ca3280_vca_lin_device> m_vca;  // U477A

	required_device<cem3310_device> m_filt_eg;  // U417
	required_device<ca3280_vca_lin_device> m_filt_eg_vca;  // U422B
	required_device<va_const_device> m_filt_freq_sh;
	// Inverting op-amp summer: LM348 + R4133 (trimmer) + R4145 + C464
	required_device<mixer_device> m_filt_freq_smr;
	required_device<filter_rc_device> m_filt_freq_smr_lpf;
	// Scale & offset resistor network: R4501 (trimmer) + R4458 + R4459 + R4502
	required_device<va_scale_offset_device> m_filt_freq_offset;
	required_device<cem3320_lpf4_device> m_vcf;  // U469

	required_ioport m_volume;  // R4529
	devcb_write8 m_volume_changed_cb;

	required_ioport m_filt_scale;  // R4133
	required_ioport m_filt_offset;  // R4501
};

INPUT_PORTS_START(prophet5_voice_trimmers)
	// Default values are based on the calibration instructions in the service manual.

	const prophet5_voice_device &voice = dynamic_cast<const prophet5_voice_device &>(owner);

	PORT_START("trimmer_volume")
	PORT_ADJUSTER(100, voice.trimmer_name_volume())
		PORT_CHANGED_MEMBER(DEVICE_SELF, FUNC(prophet5_voice_device::volume_trimmer_changed), 0);

	PORT_START("trimmer_filt_scale")
	PORT_ADJUSTER(14, voice.trimmer_name_filt_scale())
		PORT_CHANGED_MEMBER(DEVICE_SELF, FUNC(prophet5_voice_device::filter_trimmer_changed), 0)

	PORT_START("trimmer_filt_offset")
	PORT_ADJUSTER(82, voice.trimmer_name_filt_offset())
		PORT_CHANGED_MEMBER(DEVICE_SELF, FUNC(prophet5_voice_device::filter_trimmer_changed), 0)
INPUT_PORTS_END

}  // anonymous namespace

DEFINE_DEVICE_TYPE(PROPHET5_VOICE, prophet5_voice_device, "prophet5_voice", "Prophet 5 Rev 3.x voice")

prophet5_voice_device::prophet5_voice_device(
		const machine_config &mconfig,
		const char *tag,
		device_t *owner,
		device_sound_interface *filt_sum_cv,
		device_sound_interface *noise)
	: device_t(mconfig, PROPHET5_VOICE, tag, owner, 0)
	, device_sound_interface(mconfig, *this)
	, m_volume_name(util::string_format("%s TRIMMER: VOLUME", strmakeupper(basetag())))
	, m_filt_scale_name(util::string_format("%s TRIMMER: FILT SCALE", strmakeupper(basetag())))
	, m_filt_offset_name(util::string_format("%s TRIMMER: FILT OFFSET", strmakeupper(basetag())))
	, m_filt_sum_cv(filt_sum_cv)
	, m_noise(noise)
	, m_vca_eg(*this, "vca_eg")
	, m_vca(*this, "vca")
	, m_filt_eg(*this, "filt_eg")
	, m_filt_eg_vca(*this, "filt_eg_vca")
	, m_filt_freq_sh(*this, "filt_freq_sh")
	, m_filt_freq_smr(*this, "filt_freq_summer")
	, m_filt_freq_smr_lpf(*this, "filt_freq_summer_lpf")
	, m_filt_freq_offset(*this, "filt_freq_offset")
	, m_vcf(*this, "vcf")
	, m_volume(*this, "trimmer_volume")
	, m_volume_changed_cb(*this)
	, m_filt_scale(*this, "trimmer_filt_scale")
	, m_filt_offset(*this, "trimmer_filt_offset")
{
}

prophet5_voice_device::prophet5_voice_device(const machine_config &mconfig, const char *tag, device_t *owner, u32 clock)
	: prophet5_voice_device(mconfig, tag, owner, nullptr, nullptr)
{
}

void prophet5_voice_device::device_add_mconfig(machine_config &config)
{
	// Modulation and audio pipeline for a single voice. The output of each
	// stage is either a voltage or a current, as per the actual hardware.

	// *** VCA modulation ***

	// The only VCA modulation source is an ADSR envelope generator. Its voltage
	// output is converted to a current and fed to the CA3280's control input.
	CEM3310(config, m_vca_eg, RES_K(24.3), CAP_U(0.039))  // U412, R419 (1%), C427 (5%)
		.add_route(0, "amp_cc", 1.0);
	PROPHET5_CV2CC(config, "amp_cc", RES_K(3.3))  // Q410, R4496
		.add_route(0, m_vca, 1.0, ca3280_vca_lin_device::INPUT_GAIN);

	// *** Filter (VCF) control and modulation ***

	// There are two cutoff frequency control voltages (CVs). One common to all
	// voices ("filt sum cv") and a delta ("filt s/h") for each voice. The delta
	// is used when keyboard tracking is enabled. The output of both devices
	// below is a current.
	VA_CONST(config, m_filt_freq_sh).add_route(0, m_filt_freq_smr, 1.0 / RES_K(100));  // R4144 (1%)
	if (m_filt_sum_cv)
		m_filt_sum_cv->add_route(0, m_filt_freq_smr, 1.0 / RES_K(100));  // R4143 (1%)

	// The filter has its own ADSR envelope generator. Its voltage output is fed
	// to a VCA, which controls the envelope amount applied to the filter's
	// cutoff frequency. The output of the VCA is a current.
	CEM3310(config, m_filt_eg, RES_K(24.3), CAP_U(0.039))  // U417, R440 (1%), C443 (5%)
		.add_route(0, m_filt_eg_vca, 1.0);
	CA3280_VCA_LIN(config, m_filt_eg_vca, RES_K(121), VPLUS, VMINUS)  // U422B, R451 (1%)
		.configure_voltage_input(RES_K(47.5))  // R452 (1%)
		.add_route(0, m_filt_freq_smr, 1.0);

	// TODO: Emulate the "polymod" filter modulation source.

	// The control currents from all sources are summed, scaled, and low-pass-
	// filtered by U433B (LM348 op-amp) and surrounding components. The scale
	// is calibrated with the "filt scale" trimmer. The voltage output is further
	// scaled and offset by a resistor network, which is calibrated with the
	// "filt offset" trimmer. See update_filter_freq_calibration(). The
	// processed voltage is then fed to the VCF's frequency control input.
	MIXER(config, m_filt_freq_smr).add_route(0, m_filt_freq_smr_lpf, 1.0);
	FILTER_RC(config, m_filt_freq_smr_lpf)  // ~751-982 Hz.
		.add_route(0, m_filt_freq_offset, 1.0);
	VA_SCALE_OFFSET(config, m_filt_freq_offset)
		.add_route(0, m_vcf, 1.0, cem3320_lpf4_device::INPUT_FREQ);

	// *** Audio ***

	if (m_noise)
		m_noise->add_route(0, "osc_mixer", 1.0 / RES_K(100));  // R4367

	// Currents from the noise source and oscillators are summed into the VCF input.
	MIXER(config, "osc_mixer").add_route(0, m_vcf, 1.0, cem3320_lpf4_device::INPUT_AUDIO);

	// The filter's output is scaled by a non-inverting amplifier (U474B, TL082).
	constexpr double VCF_OUT_GAIN = 1.0 + RES_K(240) / RES_K(100);  // R4500 / R4499
	CEM3320_LPF4(config, m_vcf, CAP_P(150))
		.configure_resonance(RES_K(200), RES_K(51), RES_K(3), VCF_OUT_GAIN)  // R4414, R4416, R4415
		.add_route(0, m_vca, VCF_OUT_GAIN);

	CA3280_VCA_LIN(config, m_vca, RES_K(68), VPLUS, VMINUS)  // R4546
		.configure_voltage_input(RES_K(20))  // R4548
		.add_route(0, *this, 1.0);
}

ioport_constructor prophet5_voice_device::device_input_ports() const
{
	return INPUT_PORTS_NAME(prophet5_voice_trimmers);
}

void prophet5_voice_device::device_start()
{
	m_stream = stream_alloc(1, 1, machine().sample_rate());
}

void prophet5_voice_device::device_reset()
{
	update_filter_freq_calibration();
}

void prophet5_voice_device::sound_stream_update(sound_stream &stream)
{
	stream.copy(0, 0);
}

double prophet5_voice_device::volume_trimmer_r() const
{
	constexpr double R_VOL_MAX = RES_K(25);  // trimmer R4529
	return R_VOL_MAX * normalized(m_volume);
}

void prophet5_voice_device::gate_w(int state)
{
	m_vca_eg->gate_w(state);
	m_filt_eg->gate_w(state);
	LOGMASKED(LOG_GATE, "Voice: %s, gate: %d\n", tag(), state);
}

void prophet5_voice_device::filt_sh_w(double cv)
{
	m_filt_freq_sh->set_value(cv);
}

void prophet5_voice_device::filt_res_w(double cv)
{
	m_vcf->set_fixed_res_cv(cv);
	LOGMASKED(LOG_FILTER, "%s: Filter resonance CV: %f, res: %f\n", tag(), cv, m_vcf->get_res());
}

void prophet5_voice_device::filt_env_amt_w(double cc)
{
	m_filt_eg_vca->set_fixed_gain_cv(cc);
	LOGMASKED(LOG_FILTER, "%s: Filter envelope AMT CC: %f\n", tag(), cc);
}

void prophet5_voice_device::update_filter_freq_calibration()
{
	// Control voltages from multiple sources are summed and inverted by U433B
	// (LM348 op-amp) and surrounding resistors. A capacitor in the op-amp's
	// feedback turns this into an LPF.
	const double r_feedback = RES_K(162) + RES_K(50) * normalized(m_filt_scale);  // R4145 (1%) + R4133
	m_filt_freq_smr->set_output_gain(0, -r_feedback);
	m_filt_freq_smr_lpf->filter_rc_set_RC(filter_rc_device::LOWPASS, r_feedback, 0, 0, CAP_U(0.001));  // C464

	// The summed CV is further scaled and offset by a resistor network.
	constexpr double R4458 = RES_K(187);  // 1%
	constexpr double R4459 = RES_K(1.82);  // 1%
	const double r15v = RES_K(249) + RES_K(100) * normalized(m_filt_offset);  // R4502 (1%) + R4501
	const double scale = RES_VOLTAGE_DIVIDER(R4458, RES_2_PARALLEL(R4459, r15v));
	const double offset = VPLUS * RES_VOLTAGE_DIVIDER(r15v, RES_2_PARALLEL(R4458, R4459));
	m_filt_freq_offset->set_scale(scale);
	m_filt_freq_offset->set_offset(offset);

	const double lpf_freq = 1.0 / (2.0 * M_PI * r_feedback * CAP_U(0.001));
	LOGMASKED(LOG_CALIBRATION | LOG_FILTER, "%s: Filter freq CV - LPF freq: %f\n", tag(), lpf_freq);
	LOGMASKED(LOG_CALIBRATION | LOG_FILTER, "%s: Filter frequency: %f\n", tag(), m_vcf->get_freq());
}


namespace {

class prophet5_audio_device : public device_t
{
public:
	prophet5_audio_device(const machine_config &mconfig, const char *tag, device_t *owner, u32 clock = 0) ATTR_COLD;

	prophet5_voice_device *voice(int i) { return m_voices[i].target(); }

	void select_a440_w(int state);
	void a440_w(int state);

	void filt_kbd_s_w(int state);
	void wmod_filt_s_w(int state);

	void cv_w(offs_t cv_index, double cv);

	DECLARE_INPUT_CHANGED_MEMBER(mod_wheel_changed) { update_wmod_amount(); }
	DECLARE_INPUT_CHANGED_MEMBER(filter_cv_in_changed) { update_filter_fixed_cvs(); }
	DECLARE_INPUT_CHANGED_MEMBER(master_volume_changed) { update_master_volume(); }

protected:
	void device_add_mconfig(machine_config &config) override ATTR_COLD;
	void device_start() override ATTR_COLD;
	void device_reset() override ATTR_COLD;

private:
	static double eg_rate_cv(double cv);
	static double eg_sustain_cv(double cv);

	void update_wmod_amount();
	void update_wmod_routing();
	void update_filter_fixed_cvs();
	void update_voice_volume();
	void update_master_volume();

	required_device<ca3280_vca_device> m_mod_noise_vca;  // U378B
	required_device<mixer_device> m_wmod;  // R2, U374D (LM348) - W-MOD BUFFER
	required_device<va_const_device> m_filt_fixed_cvs;
	required_device<mixer_device> m_filt_cv_mixer;

	required_device_array<prophet5_voice_device, 5> m_voices;
	required_device<dac_1bit_device> m_a440;  // U315B (8253) output (pin 13)
	required_device<filter_rc_device> m_a440_lpf;  // C4183 and surrounding resistors.
	required_device<filter_rc_device> m_parasitic_filter;  // C4183's "parasitic" effect on the voice outputs.
	required_device<ca3280_vca_device> m_noise_vca;  // U430B (CA3280)
	required_device<ca3280_vca_lin_device> m_master_vol_vca;  // U479B (CA3280)

	required_ioport m_master_vol_pot;  // R113
	required_ioport m_amp_cv_in_connected;
	required_ioport m_amp_cv_in;  // J7
	required_ioport m_filter_cv_in;  // J6
	required_ioport m_mod_wheel;  // R2

	bool m_filt_kbd_s;  // U369D (CD4016) control (pin 12)
	bool m_wmod_filt_s;  // U369C (CD4016) control (pin 6)
	std::array<double, 40> m_cv;

	enum cv_type
	{
		CV_FILT_ATTACK = 0, CV_FILT_DECAY, CV_FILT_SUSTAIN, CV_FILT_RELEASE,
		CV_AMP_ATTACK, CV_AMP_DECAY, CV_AMP_SUSTAIN, CV_AMP_RELEASE,
		CV_FILT_CUTOFF, CV_FILT_ENV_AMT,
		CV_MIX_OSC_B, CV_OSC_B_PW,
		CV_MIX_OSC_A, CV_OSC_A_PW,
		CV_MIX_NOISE,
		CV_FILT_RESONANCE,
		CV_GLIDE,
		CV_LFO_FREQ, CV_WMOD_SRC_MIX,
		CV_PMOD_OSC_B, CV_PMOD_ENV_AMT,
		CV_UNISON,
		CV_SEQ_OUT,
		CV_NOT_CONNECTED_1,
		CV_OSC_1A_SH, CV_OSC_1B_SH,
		CV_OSC_2A_SH, CV_OSC_2B_SH,
		CV_OSC_3A_SH, CV_OSC_3B_SH,
		CV_OSC_4A_SH, CV_OSC_4B_SH,
		CV_OSC_5A_SH, CV_OSC_5B_SH,
		CV_FILT_1_SH, CV_FILT_2_SH, CV_FILT_3_SH, CV_FILT_4_SH, CV_FILT_5_SH,
		CV_NOT_CONNECTED_2,
	};
};

}  // anonymous namespace

DEFINE_DEVICE_TYPE(PROPHET5_AUDIO, prophet5_audio_device, "prophet5_audio", "Prophet 5 Rev 3.x audio circuits")

prophet5_audio_device::prophet5_audio_device(const machine_config &mconfig, const char *tag, device_t *owner, u32 clock)
	: device_t(mconfig, PROPHET5_AUDIO, tag, owner, clock)
	, m_mod_noise_vca(*this, "mod_noise_vca")
	, m_wmod(*this, "wmod")
	, m_filt_fixed_cvs(*this, "filt_fixed_cvs")
	, m_filt_cv_mixer(*this, "filt_cv_mixer")
	, m_voices(*this, "voice_%u", 0U)
	, m_a440(*this, "a440_generator")
	, m_a440_lpf(*this, "a440_lpf")
	, m_parasitic_filter(*this, "parasitic_filter")
	, m_noise_vca(*this, "noise_vca")
	, m_master_vol_vca(*this, "master_volume_vca")
	, m_master_vol_pot(*this, ":pot_volume")
	, m_amp_cv_in_connected(*this, ":cv_in_amp_connected")
	, m_amp_cv_in(*this, ":cv_in_amp")
	, m_filter_cv_in(*this, ":cv_in_filter")
	, m_mod_wheel(*this, ":wheel_mod")
	, m_filt_kbd_s(false)
	, m_wmod_filt_s(false)
{
	std::fill(m_cv.begin(), m_cv.end(), -1);
}

void prophet5_audio_device::device_add_mconfig(machine_config &config)
{
	// *** Modulation ***

	// Pink noise modulation source.
	// The output of the noise IC is offset, scaled, inverted and low-pass-
	// filtered by U374B (LM348 op-amp) and surrounding passives. The
	// amplification will cause the signal to get clipped on the positive side.
	// That signal is then fed to a VCA, which controls the amount of modulation
	// by the noise source.
	// TODO: Implement the "noise bal" trimmer.
	// TODO: Intensity seems to max out early in the mod wheel's range. Figure out why.

	// Offset, scale, inversion, clipping and centering (in lieu of the trimmer)
	// are all handled by "mod_noise_buffer". Relevant calculations:
	constexpr double R393 = RES_K(100);
	constexpr double R394 = RES_K(300);
	constexpr double R395 = RES_K(47);
	constexpr double OPAMP_MAX = VPLUS - 2;  // Typical LM348 output limit on a +/-15V supply.
	constexpr double NOISE_MIN = std::clamp((0.0 / R395 + VPLUS / R394) * -R393, -OPAMP_MAX, OPAMP_MAX);
	constexpr double NOISE_MAX = std::clamp((VMINUS / R395 + VPLUS / R394) * -R393, -OPAMP_MAX, OPAMP_MAX);

	MM5837_STREAM(config, "mod_noise")  // U375
		.set_vdd(VMINUS)
		.add_route(0, "mod_noise_buffer", VMINUS);
	VA_SCALE_OFFSET(config, "mod_noise_buffer")  // U374B (LM348)
		.set_scale((NOISE_MAX - NOISE_MIN) / VMINUS)
		.set_offset(-(NOISE_MAX - NOISE_MIN) / 2.0)
		.add_route(0, "mod_noise_lpf", 1.0);
	FILTER_RC(config, "mod_noise_lpf")  // ~159 Hz
		.set_lowpass(R393, CAP_U(0.01))  // C372
		.add_route(0, m_mod_noise_vca, 1.0);
	CA3280_VCA(config, m_mod_noise_vca)  // U378B
		.configure_input_divider(RES_K(20), RES_R(330))  // R3119, R3120
		.add_route(0, m_wmod, 1.0);

	// TODO: Implement LFO modulation source.

	// The output currents from the noise and LFO VCAs are summed and converted
	// to a voltage by the mod wheel and R3113 (see update_wmod_amount()). The
	// voltage is buffered by U374D (LM348) and distributed to all "sum CV"
	// generators (see update_wmod_routing()).
	MIXER(config, m_wmod).add_route(0, m_filt_cv_mixer, 1.0);

	// Filter cutoff frequency "sum CV" generator.
	// Mixes multiple sources of cutoff frequency CVs: firmware-controlled CV,
	// external CV in (those two are combined in m_filt_fixed_cvs), glide out CV
	// (when keyboard tracking is enabled), and wheel modulation (when enabled
	// for the filter). The summed CV is then offset, low-pass-filtered, and
	// distributed to all voices.
	constexpr double R357 = RES_K(100);  // 1%
	VA_CONST(config, m_filt_fixed_cvs)
		.add_route(0, m_filt_cv_mixer, 1.0);
	MIXER(config, m_filt_cv_mixer)
		.add_route(0, "filt_cv_offset", -RES_K(100));  // R354 (1%), voltage
	VA_SCALE_OFFSET(config, "filt_cv_offset")
		.set_scale(1.0 / RES_K(100))  // R355 (1%)
		.set_offset(VPLUS / RES_K(261))  // R356 (1%)
		.add_route(0, "filt_sum_cv", -R357);
	auto &filt_sum_cv = FILTER_RC(config, "filt_sum_cv")  // ~159 Hz
		.set_lowpass(R357, CAP_U(0.01));  // C363


	// *** Audio ***

	// The values in each stream represent voltages or currents, as per the real
	// hardware. This scaler converts the final voltage to audio within the
	// range [-1, 1].
	constexpr double VOLTAGE_TO_AUDIO_SCALER = 0.20;

	// White noise generator. A single source is used for all voices.
	// The input level to the CA3280 is >10x above its linear range, but this
	// doesn't matter for noise.
	constexpr double R4131 = RES_K(200);
	constexpr double R4132 = RES_K(10);
	MM5837_STREAM(config, "noise")
		.set_vdd(VMINUS)
		.add_route(0, "noise_hpf", VMINUS);
	FILTER_RC(config, "noise_hpf")  // ~7.6 Hz HPF
		.set_rc(filter_rc_device::HIGHPASS, R4131 + R4132, 0, 0, CAP_U(0.1))  // C458
		.add_route(0, m_noise_vca, 1.0);
	CA3280_VCA(config, m_noise_vca)
		.configure_input_divider(R4131, R4132)  // Scaled amplitude: +/- ~0.36V.
		.configure_voltage_output(RES_K(10));  // R4129

	// The 5 voices. The output of each voice device is a current, but it is
	// converted to a voltage by setting the output gain on each voice, in
	// update_voice_volume().
	for (int i = 0; i < m_voices.size(); ++i)
	{
		PROPHET5_VOICE(config, m_voices[i], &filt_sum_cv, m_noise_vca)
			.add_route(0, "voice_summer", 1.0)
			.add_route(0, "parasitic_filter_mixer", 1.0);
		m_voices[i]->volume_changed_cb().set([this] (u8 data) { update_voice_volume(); });
	}

	// A440 tone generator. The LPF's parameters can vary, and are computed in
	// update_voice_volume().
	DAC_1BIT(config, m_a440)
		.set_output_range(0, VCC)
		.add_route(0,  m_a440_lpf, 1.0);
	FILTER_RC(config, m_a440_lpf)  // ~207-216 Hz LPF
		.add_route(0, "voice_summer", 1.0);

	// The A440 LPF will also filter the voice outputs to a small extent. Its
	// effect will be that of a high shelving filter. This is emulated by routing
	// the voice outputs to a "parasitic" HPF with the same RC values as the LPF.
	// The HPF's output is scaled and subtracted from the rest of the mix in
	// "voice_summer", to produce the shelving effect. The HPF's parameters and
	// its (negative) output gain are computed in update_voice_volume().
	MIXER(config, "parasitic_filter_mixer").add_route(0, m_parasitic_filter, 1.0);
	FILTER_RC(config, m_parasitic_filter).add_route(0, "voice_summer", 1.0);

	// Passive mixing of all voices and the tone generator, followed by a buffer
	// (non-inverting TL082 op-amp, U480B). The gain of each input is computed
	// in update_voice_volume().
	MIXER(config, "voice_summer").add_route(0, m_master_vol_vca, 1.0);

	// The master volume is controlled by a VCA. Its control current is set by
	// the master volume knob and, if connected, the external AMPLIFIER CV IN
	// input (J7). See update_master_volume().
	CA3280_VCA_LIN(config, m_master_vol_vca, RES_K(68), VPLUS, VMINUS)  // R4561
		.configure_voltage_input(RES_K(15))  // R4564
		.configure_voltage_output(RES_K(20))  // R4562
		.add_route(0, "dcblock", 1.0);

	// Output stage.
	FILTER_RC(config, "dcblock")  // 0.7 Hz HPF
		.set_rc(filter_rc_device::HIGHPASS, RES_K(100), 0, 0, CAP_U(2.2))  // R4543, C4189
		.add_route(0, "audio_out", VOLTAGE_TO_AUDIO_SCALER);
	SPEAKER(config, "audio_out").front_center();  // J1, buffered by U481 (NE5534 op-amp)
}

void prophet5_audio_device::device_start()
{
	save_item(NAME(m_filt_kbd_s));
	save_item(NAME(m_wmod_filt_s));
	save_item(NAME(m_cv));
}

void prophet5_audio_device::device_reset()
{
	for (int i = 0; i < m_cv.size(); ++i)
		cv_w(i, 0);

	update_wmod_amount();
	update_wmod_routing();
	update_filter_fixed_cvs();
	update_voice_volume();
	update_master_volume();
}

void prophet5_audio_device::select_a440_w(int state)
{
	// The A440 tone is enabled / disabled by a collection of CD4016 switches.
	// U459C controls whether the tone makes it to the mixer, U460C grounds
	// the mixer input when the tone is disabled to remove any signal that
	// bleeds through, and U461C inverts the control signal for U460C.
	m_a440->set_output_gain(0, state ? 1.0 : 0.0);
}

void prophet5_audio_device::a440_w(int state)
{
	m_a440->write(state);
}

void prophet5_audio_device::filt_kbd_s_w(int state)
{
	m_filt_kbd_s = bool(state);
	LOGMASKED(LOG_PROG_LATCH, "filt_kbd_s = %d\n", m_filt_kbd_s);
	update_filter_fixed_cvs();
}

void prophet5_audio_device::wmod_filt_s_w(int state)
{
	m_wmod_filt_s = bool(state);
	LOGMASKED(LOG_PROG_LATCH, "wmod_filt_s = %d\n", m_wmod_filt_s);
	update_wmod_routing();
}

void prophet5_audio_device::cv_w(offs_t cv_index, double cv)
{
	constexpr const char *CV_NAMES[40] =
	{
		"FILT ATTACK", "FILT DECAY", "FILT SUSTAIN", "FILT RELEASE",
		"AMP ATTACK", "AMP DECAY", "AMP SUSTAIN", "AMP RELEASE",
		"FILT CUTOFF", "FILT ENV AMT", "MIX OSC B", "OSC B PW",
		"MIX OSC A", "OSC A PW", "MIX NOISE", "FILT RESONANCE",
		"GLIDE", "LFO FREQ", "WMOD SRC MIX", "PMOD OSC B",
		"PMOD ENV AMT", "UNISON", "SEQ CV OUT", "NOT CONNECTED 1",
		"OSC 1A S/H", "OSC 1B S/H", "OSC 2A S/H", "OSC 2B S/H",
		"OSC 3A S/H", "OSC 3B S/H", "OSC 4A S/H", "OSC 4B S/H",
		"OSC 5A S/H", "OSC 5B S/H", "FILT 1 S/H", "FILT 2 S/H",
		"FILT 3 S/H", "FILT 4 S/H", "FILT 5 S/H", "NOT CONNECTED 2",
	};

	if (cv == m_cv[cv_index])
		return;

	m_cv[cv_index] = cv;
	LOGMASKED(LOG_CV, "Set CV %d (%s): %f V)\n", cv_index, CV_NAMES[cv_index], cv);

	switch (cv_index)
	{
		case CV_WMOD_SRC_MIX:
			m_mod_noise_vca->set_fixed_gain_cv(cv2cc(cv, RES_K(8.2)));  // Q307, R3116
			break;

		case CV_MIX_NOISE:
			m_noise_vca->set_fixed_gain_cv(cv2cc(cv, RES_K(75)));  // Q305, R327
			break;

		case CV_UNISON:
			if (m_filt_kbd_s)
				update_filter_fixed_cvs();
			break;

		case CV_FILT_CUTOFF: update_filter_fixed_cvs(); break;
		case CV_FILT_1_SH: m_voices[0]->filt_sh_w(cv); break;
		case CV_FILT_2_SH: m_voices[1]->filt_sh_w(cv); break;
		case CV_FILT_3_SH: m_voices[2]->filt_sh_w(cv); break;
		case CV_FILT_4_SH: m_voices[3]->filt_sh_w(cv); break;
		case CV_FILT_5_SH: m_voices[4]->filt_sh_w(cv); break;
		case CV_FILT_RESONANCE:
			for (prophet5_voice_device *v : m_voices)
				v->filt_res_w(cv);
			break;

		case CV_FILT_ENV_AMT:
		{
			// The control current is split (approximately) equally across the
			// env amount VCAs of the 5 voices.
			const double cc = cv2cc(cv, RES_K(5.1)) / double(m_voices.size());  // Q301, R321
			for (prophet5_voice_device *v : m_voices)
				v->filt_env_amt_w(cc);
			break;
		}
		case CV_FILT_ATTACK:
		{
			const double eg_cv = eg_rate_cv(cv);
			for (prophet5_voice_device *v : m_voices)
				v->filt_eg()->attack_w(eg_cv);
			break;
		}
		case CV_FILT_DECAY:
		{
			const double eg_cv = eg_rate_cv(cv);
			for (prophet5_voice_device *v : m_voices)
				v->filt_eg()->decay_w(eg_cv);
			break;
		}
		case CV_FILT_SUSTAIN:
		{
			const double eg_cv = eg_sustain_cv(cv);
			for (prophet5_voice_device *v : m_voices)
				v->filt_eg()->sustain_w(eg_cv);
			break;
		}
		case CV_FILT_RELEASE:
		{
			const double eg_cv = eg_rate_cv(cv);
			for (prophet5_voice_device *v : m_voices)
				v->filt_eg()->release_w(eg_cv);
			break;
		}

		case CV_AMP_ATTACK:
		{
			const double eg_cv = eg_rate_cv(cv);
			for (prophet5_voice_device *v : m_voices)
				v->vca_eg()->attack_w(eg_cv);
			break;
		}
		case CV_AMP_DECAY:
		{
			const double eg_cv = eg_rate_cv(cv);
			for (prophet5_voice_device *v : m_voices)
				v->vca_eg()->decay_w(eg_cv);
			break;
		}
		case CV_AMP_SUSTAIN:
		{
			const double eg_cv = eg_sustain_cv(cv);
			for (prophet5_voice_device *v : m_voices)
				v->vca_eg()->sustain_w(eg_cv);
			break;
		}
		case CV_AMP_RELEASE:
		{
			const double eg_cv = eg_rate_cv(cv);
			for (prophet5_voice_device *v : m_voices)
				v->vca_eg()->release_w(eg_cv);
			break;
		}
	}
}

double prophet5_audio_device::eg_rate_cv(double cv)
{
	// The rate CVs (attack, decay, release) for both EGs are converted from
	// ~[0V, 10V] to ~[-0.283V, 0.020V], before being supplied to the CEM3310s.
	// The tolerance of all resistors is 1%.
	//
	//  CV-----R(24.3K)----+--- CEM3310 rate CV input.
	//                     |
	// -5V-----R(13K)------+
	//                     |
	//  GND----R(806)------+

	constexpr double SCALE = RES_VOLTAGE_DIVIDER(RES_K(24.3), RES_2_PARALLEL(RES_K(13), RES_R(806)));
	constexpr double OFFSET = -5 * RES_VOLTAGE_DIVIDER(RES_K(13), RES_2_PARALLEL(RES_K(24.3), RES_R(806)));
	return cv * SCALE + OFFSET;
}

double prophet5_audio_device::eg_sustain_cv(double cv)
{
	// The sustain CVs for both EGs are halved to ~[0V, 5V] by resistive
	// dividers before being fed to the CEM3310s.
	return cv * RES_VOLTAGE_DIVIDER(RES_K(4.75), RES_K(4.75));  // both 1%
}

void prophet5_audio_device::update_wmod_amount()
{
	// The two VCAs are current sources.
	//
	// Noise VCA----+-----+------+
	//              |     |      |
	// LFO VCA------+   R3113    R2---BUFFER---[multiple "sum CV" circuits]
	//                    |      |
	//                   GND    GND

	constexpr double R2 = RES_K(100);  // mod wheel
	constexpr double R3113 = RES_K(10);

	// Compute the gain that converts the sum of the VCA currents to the voltage
	// at the buffer.
	const double r2_bottom = normalized(m_mod_wheel) * R2;
	const double r2_top = R2 - r2_bottom;
	const double i2v = RES_2_PARALLEL(R3113, R2) * RES_VOLTAGE_DIVIDER(r2_top, r2_bottom);

	m_wmod->set_output_gain(0, i2v);
	LOGMASKED(LOG_WHEEL, "Mod wheel: %d, I2V: %f\n", m_mod_wheel->read(), i2v);
}

void prophet5_audio_device::update_wmod_routing()
{
	const double filt_gain = m_wmod_filt_s ? (1.0 / RES_K(13.3)) : 0.0;  // R399 (1%)
	m_wmod->set_route_gain(0, m_filt_cv_mixer, 0, filt_gain);
	LOGMASKED(LOG_WHEEL, "Mod wheel routing - filter: %f\n", filt_gain);
}

void prophet5_audio_device::update_filter_fixed_cvs()
{
	// Multiple cutoff frequency sources are summed and inverted by U367A (
	// LM348, configured as an inverting mixer) and surrounding resistors.
	// TODO: Glide amount is not yet emulated. The "glide out" CV tracks the
	// "unison" CV for now, regardless of glide amount.
	const double i_cv_in = MAX_CV_IN * normalized(m_filter_cv_in) / RES_K(100);  // R374 (1%)
	const double i_ctf_cv = m_cv[CV_FILT_CUTOFF] / RES_K(100);  // R352 (1%)
	const double i_glide_out_cv = m_filt_kbd_s ? (m_cv[CV_UNISON] / RES_K(100)) : 0.0;  // R372 (1%)
	m_filt_fixed_cvs->set_value(i_cv_in + i_ctf_cv + i_glide_out_cv);
}

void prophet5_audio_device::update_voice_volume()
{
	// The 5 voices and the A440 tone are mixed passively and buffered by an
	// op-amp. The diagram below shows 2 of the 5 voices, and the A440 tone.
	//
	// Voice 1 VCA---+---Rvoice-----+
	//               |              |
	//              Rvol trimmer    |
	//               |              |
	//              GND             |
	//                              |
	// Voice 2 VCA---+---Rvoice-----+
	//               |              |
	//              Rvol trimmer    |       Voice VCAs are current sources.
	//               |              |       A440 is a voltage source.
	//              GND             |
	//                              |
	// A440--- R4498--+--R4519------+------ BUFFER --- Master VCA
	//                |                 (U480B, TL082)
	//              C4183
	//                |
	//               GND
	//
	// Because of the passive mixing, there are multiple interactions between
	// the sub-circuits above. For instance:
	// * The effective R of the A440 RC LPF is ~22-26% smaller than the apparent
	//   value of R4498.
	// * Adjusting the volume trimmer of a voice will also affect the volume of
	//   the other voices and the A440 tone.
	// * Adjusting the volume trimmers affects the cutoff frequency of the A440 LPF.
	// * The A440 LPF will also act as a high shelving filter for the voice
	//   outputs, attenuating high frequencies by a small amount.
	//
	// In analyzing this circuit, keep in mind that the voice VCAs are current
	// sources, while A440 is a voltage source.
	//
	// To find the voltage at the buffer, just sum up the contributions from
	// each source. To find the contribution from each source, treat all other
	// current sources as disconnected, and all other voltage sources as
	// grounded. Then calculate the voltage at the buffer's input due to the
	// source being examined.
	//
	// Similarly, to find the effective R of the RC circuit, treat all current
	// sources as disconnected and all voltage sources as grounded, and compute
	// the resistance to ground from the non-grounded side of the capacitor.

	constexpr double R_VOICE = RES_K(39);  // R4569, R4568, R4567, R4566, R4565
	constexpr double R4498 = RES_K(10);
	constexpr double R4519 = RES_K(20);
	constexpr double R_A440 = R4498 + R4519;
	constexpr double C_A440 = CAP_U(0.1);  // C4183

	std::array<double, 5> r_vol;  // Resistance of volume trimmers.
	std::array<double, 5> r_input;  // Resistance from the buffer's input to each voice.
	for (int i = 0; i < m_voices.size(); ++i)
	{
		r_vol[i] = m_voices[i]->volume_trimmer_r();
		r_input[i] = R_VOICE + r_vol[i];
	}

	// Given each voice, this is the resistance to ground, from the buffer's
	// input to all other voices and the grounded A440 source.
	const std::array<double, 5> r_other =
	{
		RES_5_PARALLEL(r_input[1], r_input[2], r_input[3], r_input[4], R_A440),
		RES_5_PARALLEL(r_input[0], r_input[2], r_input[3], r_input[4], R_A440),
		RES_5_PARALLEL(r_input[0], r_input[1], r_input[3], r_input[4], R_A440),
		RES_5_PARALLEL(r_input[0], r_input[1], r_input[2], r_input[4], R_A440),
		RES_5_PARALLEL(r_input[0], r_input[1], r_input[2], r_input[3], R_A440),
	};

	LOGMASKED(LOG_CALIBRATION, "Voice volume adjusted. New gains:\n");
	for (int i = 0; i < m_voices.size(); ++i)
	{
		// Calculate contribution from each voice. Note that `gain` also does
		// current-to-voltage conversion (recall that voices are current
		// sources). So its value can be in the thousands.
		const double gain = r_vol[i] * r_other[i] / (r_vol[i] + R_VOICE + r_other[i]);
		m_voices[i]->set_output_gain(0, gain);
		LOGMASKED(LOG_CALIBRATION, "  - Voice %d: %f\n", i, gain);
	}

	// Calculate contribution from the A440 tone.
	const double a440_r_other = RES_5_PARALLEL(r_input[0], r_input[1], r_input[2], r_input[3], r_input[4]);
	const double a440_gain = RES_VOLTAGE_DIVIDER(R_A440, a440_r_other);
	m_a440_lpf->set_output_gain(0, a440_gain);
	LOGMASKED(LOG_CALIBRATION, "  - A440   : %f\n", a440_gain);

	// Calculate parameters for the A440 tone's LPF.
	// The RC values are the same for the parasitic HPF.
	const double a440_rc_r_eq = RES_2_PARALLEL(R4498, R4519 + a440_r_other);
	const double a440_rc_freq = 1.0 / (2.0 * M_PI * a440_rc_r_eq * C_A440);
	m_a440_lpf->filter_rc_set_RC(filter_rc_device::LOWPASS, a440_rc_r_eq, 0, 0, C_A440);
	m_parasitic_filter->filter_rc_set_RC(filter_rc_device::HIGHPASS, a440_rc_r_eq, 0, 0, C_A440);
	LOGMASKED(LOG_CALIBRATION, "A440 LPF - Req: %f, freq: %f\n", a440_rc_r_eq, a440_rc_freq);

	// Calculate the proportion of the parasitic HPF's output that should be
	// subtracted from the mixed voice signal. This subtraction will produce the
	// high shelving effect of the A440 LPF's on the voice signal.

	// That proportion is the maximum attenuation produced by the shelving
	// filter. To compute it, find the impedance on the buffer's input for
	// infinite-frequency AC (C_A440 treated as a short to ground) and for DC (
	// C_A440 treated as disconnected, keeping in mind that C_A440 is parallel
	// to R4498. Then, compute the proportion based on the ratio of those
	// impedances.

	const double r_dc = RES_6_PARALLEL(r_input[0], r_input[1], r_input[2], r_input[3], r_input[4], R4519 + R4498);
	const double r_ac = RES_6_PARALLEL(r_input[0], r_input[1], r_input[2], r_input[3], r_input[4], R4519);
	const double parasitic_hpf_gain = -(1.0 - r_ac / r_dc);
	m_parasitic_filter->set_output_gain(0, parasitic_hpf_gain);
	LOGMASKED(LOG_CALIBRATION, "Parasitic HPF gain: %f\n", parasitic_hpf_gain);
}

void prophet5_audio_device::update_master_volume()
{
	//               5V
	//               |
	//              R345
	//               |
	//               \  <-- Normally closed. Open when AMP CV IN is connected.
	//               +-----AMP CV IN
	//               |
	// Volume knob: R113----+---- BUFFER (U480A, TL082) --- Master Volume CV
	//               |      |
	//               |    R4535
	//               |      |
	//              GND    GND

	constexpr double R113_MAX = RES_K(100);  // Volume pot max value.
	constexpr double R4535 = RES_K(100);  // Makes the volume taper non-linear.

	const bool cv_in_connected = BIT(m_amp_cv_in_connected->read(), 0);
	const double volume = normalized(m_master_vol_pot);
	double max_vol_cv = 0;
	double vol_cv = 0;

	if (m_master_vol_pot->read() > 0)
	{
		double r_top = 0;
		if (cv_in_connected)
		{
			r_top = 0;
			max_vol_cv = MAX_CV_IN * normalized(m_amp_cv_in);
		}
		else
		{
			r_top = RES_R(100);  // R345
			max_vol_cv = VCC;
		}

		const double r_vol_bottom = R113_MAX * volume;
		const double r_vol_top = R113_MAX - r_vol_bottom;
		vol_cv = max_vol_cv * RES_VOLTAGE_DIVIDER(r_top + r_vol_top, RES_2_PARALLEL(r_vol_bottom, R4535));
	}

	const double vol_cc = cv2cc(vol_cv, RES_K(4.7));  // Q411, R4542
	m_master_vol_vca->set_fixed_gain_cv(vol_cc);

	LOGMASKED(LOG_CV, "Master volume changed. Pot: %f, CV in connected: %d, Max CV: %f, VOL CV: %f, VOL CC: %f\n",
			  volume, cv_in_connected, max_vol_cv, vol_cv, vol_cc);
}


namespace {

constexpr const char AUDIO_TAG[] = "prophet5_audio";

class prophet5_state : public driver_device
{
public:
	static constexpr feature_type unemulated_features() { return feature::TAPE; }

	prophet5_state(const machine_config &mconfig, device_type type, const char *tag) ATTR_COLD;

	void prophet5rev30(machine_config &config) ATTR_COLD;

	DECLARE_INPUT_CHANGED_MEMBER(record_changed);
	DECLARE_INPUT_CHANGED_MEMBER(gate_in_changed);
	DECLARE_INPUT_CHANGED_MEMBER(pot_adjusted);
	DECLARE_INPUT_CHANGED_MEMBER(dac_trimmer_adjusted);
	DECLARE_INPUT_CHANGED_MEMBER(adc_trimmer_adjusted);
	DECLARE_INPUT_CHANGED_MEMBER(cv_in_changed);
	DECLARE_INPUT_CHANGED_MEMBER(seq_trimmer_adjusted);

protected:
	void machine_start() override ATTR_COLD;
	void machine_reset() override ATTR_COLD;

private:
	static double i_bias(const required_ioport &rp, double rp_max, double r, double v);

	void switch_w(u8 data);
	u8 switch_r();
	u8 misc_r();
	u8 adc_r();

	void led_sink_w(u8 data);
	void led_drive_w(u8 data);
	void led_update_w(offs_t offset, u8 data);

	void update_sh();
	void update_vdac();
	void update_vmux();
	void mux_abc_w(u8 data);
	void sh_mux_inh_w(u8 data);
	void dac_w(offs_t offset, u8 data);
	void pot_mux_w(u8 data);

	void update_gate5();
	void latch_gate5_w(int state);
	void clr_int_w(u8 data);
	TIMER_DEVICE_CALLBACK_MEMBER(gate_in_delay_elapsed);

	void update_nvram_record();

	void memory_map(address_map &map) ATTR_COLD;
	void io_map(address_map &map) ATTR_COLD;

	required_device<prophet5_audio_device> m_audio;
	required_device<z80_device> m_maincpu;  // U311
	required_device<ttl7474_device> m_tune_ff;  // U322A
	required_device<timer_device> m_gate_in_delay;  // R311, C316, U331A
	required_device<pwm_display_device> m_led_matrix_pwm;
	required_device<pwm_display_device> m_digit_pwm;
	memory_view m_nvram_write_view;
	required_ioport_array<16> m_switches;
	required_ioport m_record;
	required_ioport m_release_footswitch;
	required_ioport m_gate_in;
	required_ioport m_gate_in_connected;
	required_ioport m_test_points;
	required_ioport_array<24> m_pots;
	required_ioport m_dac_gain;
	required_ioport m_adc_gain;
	required_ioport m_seq_cv_in;
	required_ioport m_seq_offset;
	required_ioport m_seq_scale;
	output_finder<> m_tune_mux_select;
	output_finder<> m_tuning;
	std::vector<std::vector<output_finder<>>> m_leds;

	u8 m_switch_row = 0;  // U212 input (CD4514 decoder).
	u8 m_mux_abc = 0;  // U338 (CD4174 latch): Q3, Q2, Q5 (MSbit to LSbit).
	u8 m_sh_mux_inh = 0x1f;  // U339 (CD4174): Q4, Q1, Q3, Q2, Q5.
	bool m_seq_cv_enabled = false;  // U339 (CD4174): Q0.
	u16 m_dac_latch = 0;  // 2 x CD4174 (U336, U337) + 2 x CD4013 (U342A, B).
	u8 m_pot_mux_abc = 0;  // U211 (CD4174): Q1, Q5, Q0 (MSbit to LSbit).
	u8 m_pot_mux_inh = 0x07;  // U211 (CD4174): Q3, Q2, Q4.
	double m_vdac = 0;
	double m_adc_vmux = 0;
	double m_adc_vref = 0;
	bool m_tune_counter_out = false;  // 8253 (U315) counter 0 output.
	bool m_latch_gate5 = false;  // U340 (CD4174) output Q4 (pin 12).
	bool m_ext_gate5 = false;  // U330B (CD4013) output Q (pin 13).

	static inline constexpr double VPLUS = 15.0;
	static inline constexpr double VMINUS = -15.0;
};

prophet5_state::prophet5_state(const machine_config &mconfig, device_type type, const char *tag)
	: driver_device(mconfig, type, tag)
	, m_audio(*this, AUDIO_TAG)
	, m_maincpu(*this, "maincpu")
	, m_tune_ff(*this, "tune_ff")
	, m_gate_in_delay(*this, "gate_in_delay")
	, m_led_matrix_pwm(*this, "led_matrix_pwm")
	, m_digit_pwm(*this, "led_digit_pwm")
	, m_nvram_write_view(*this, "nvram_write_view")
	, m_switches(*this, "switch_row_%u", 0U)
	, m_record(*this, "record")
	, m_release_footswitch(*this, "release_footswitch")
	, m_gate_in(*this, "gate_in")
	, m_gate_in_connected(*this, "gate_in_connected")
	, m_test_points(*this, "test_points")
	, m_pots(*this, "pot_%u", 0U)
	, m_dac_gain(*this, "trimmer_dac_gain")
	, m_adc_gain(*this, "trimmer_adc_gain")
	, m_seq_cv_in(*this, "cv_in_seq")
	, m_seq_offset(*this, "trimmer_seq_offset")
	, m_seq_scale(*this, "trimmer_seq_scale")
	, m_tune_mux_select(*this, "tune_mux_select")
	, m_tuning(*this, "tuning")
{
	static constexpr const char *LED_NAMES[8][5] =
	{
		{"osc_a_sqr",  "pmod_freq_a", "wmod_freq_a", "ps1", "record"},
		{"osc_a_saw",  "pmod_pw_a",   "wmod_freq_b", "ps2", "unused_1"},
		{"osc_a_sync", "pmod_filt",   "wmod_pw_a",   "ps3", "a_440"},
		{"osc_b_saw",  "lfo_saw",     "wmod_pw_b",   "ps4", "tune"},
		{"osc_b_tri",  "lfo_tri",     "wmod_filt",   "ps5", "to_cass"},
		{"osc_b_sqr",  "lfo_sqr",     "osc_b_lo",    "ps6", "from_cass"},
		{"osc_b_kbd",  "filt_kbd",    "unused_2",    "ps7", "unused_3"},
		{"unison",     "release",     "unused_4",    "ps8", "preset"},
	};

	for (int y = 0; y < 8; ++y)
	{
		m_leds.push_back(std::vector<output_finder<>>());
		for (int x = 0; x < 5; ++x)
			m_leds[y].push_back(output_finder<>(*this, std::string("led_") + LED_NAMES[y][x]));
	}
}

// Computes the current through resistor R, from the junction of the resistors
// towards V. Rp1 and Rp2 are the two sides of a single potentiometer.
//
//  V+ --- Rp1 --*-- Rp2 --- V-
//               |
//               R
//               |
//               V
double prophet5_state::i_bias(const required_ioport &rp, double rp_max, double r, double v)
{
	const double rp1 = rp_max * normalized(rp);
	const double rp2 = rp_max - rp1;
	// Compute voltage at the junction of all resistors.
	const double vx = (r * rp1 * VMINUS + r * rp2 * VPLUS + rp1 * rp2 * v) / (r * rp1 + r * rp2 + rp1 * rp2);
	return (vx - v) / r;
}

void prophet5_state::switch_w(u8 data)
{
	m_switch_row = data & 0x0f;
}

u8 prophet5_state::switch_r()
{
	const u8 pressed = m_switches[m_switch_row]->read();
	if (pressed)
		LOGMASKED(LOG_SWITCHES, "Switches - row: %d, pressed: %02x\n", m_switch_row, pressed);
	return pressed;
}

u8 prophet5_state::misc_r()
{
	const u8 d0 = 1;  // Cassette in. Will settle to 1 when there is no cassette input.
	const u8 d1 = BIT(m_release_footswitch->read(), 0);
	const u8 d2 = m_tune_ff->output_comp_r();
	const u8 d3 = BIT(m_gate_in_connected->read(), 0); // External gate enabled (input connected).
	const u8 d4 = BIT(m_record->read(), 0);  // Record enabled (NVRAM write protect disabled).
	const u8 d5 = m_tune_counter_out ? 1 : 0;
	return (d5 << 5) | (d4 << 4) | (d3 << 3) | (d2 << 2) | (d1 << 1) | d0;
}

u8 prophet5_state::adc_r()
{
	// The ADC consists of two comparators (U365C,D, LM339). A network of
	// resistors and diodes create the reference inputs to the two comparators,
	// by adding and subtracting 34mV to the ADC reference. The ADC reference is
	// nominally half the DAC output voltage.
	const u8 d0 = (m_adc_vmux < m_adc_vref - 0.034) ? 1 : 0;  // ADC LO
	const u8 d1 = (m_adc_vmux > m_adc_vref + 0.034) ? 1 : 0;  // ADC HI
	if (d0 || d1)
	{
		LOGMASKED(LOG_ADC, "ADC: Vmux: %f, Vref: %f - lo: %d,  hi: %d\n",
				  m_adc_vmux, m_adc_vref, d0, d1);
	}

	const u8 test_points = m_test_points->read();
	const u8 d2 = BIT(test_points, 1);  // TP301
	const u8 d3 = BIT(test_points, 4);  // TP304
	const u8 d4 = 0;  // Connected to GND.
	const u8 d5 = BIT(test_points, 6);  // TP306

	return (d5 << 5) | (d4 << 4) | (d3 << 3) | (d2 << 2) | (d1 << 1) | d0;
}

void prophet5_state::led_sink_w(u8 data)
{
	// The full LED matrix size is 8x7. Columns 0-4 control individual LEDs
	// (m_led_matrix_pwm), and columns 5 and 6 control the "bank" and "program"
	// 7-segment digit displays, respectively (m_digit_pwm). Only the first 7
	// rows are used for the digit displays.

	m_led_matrix_pwm->write_mx(data & 0x1f);

	// Using write_my() because video/pwm.cpp assumes the selected digit is in
	// the row.
	m_digit_pwm->write_my((data >> 5) & 0x03);
}

void prophet5_state::led_drive_w(u8 data)
{
	m_led_matrix_pwm->write_my(data);
	m_digit_pwm->write_mx(data & 0x7f);
}

void prophet5_state::led_update_w(offs_t offset, u8 data)
{
	m_leds[offset & 0x3f][offset >> 6] = data;
}

void prophet5_state::update_sh()
{
	if ((m_sh_mux_inh & 0x1f) == 0x1f)
		return;  // Exit early if no S&H is selected.

	for (int i = 0; i < 5; ++i)
		if (!BIT(m_sh_mux_inh, i))  // Active low.
			m_audio->cv_w(8 * i + m_mux_abc, m_vdac);
}

void prophet5_state::update_vdac()
{
	// CVs are produced by DAC71-CSB-I, a 16-bit, unipolar, current output DAC.
	// Some units shipped with the DAC71-CSB-V (voltage output), with
	// corresponding changes to the DAC output buffer.

	constexpr double I_FS_NOMINAL = 1.99997e-3;  // From the datasheet.
	constexpr double DAC_RF = RES_K(5);  // Internal DAC71-CSB feedback resistor.

	// The contents of the DAC latch are inverted by U343, U344 and U345
	// (CD4049). While the DAC71 is a 16-bit DAC, only the 14 MSbits are used.
	// The 2 LSbits are pulled high.
	const u16 dac_input = (~m_dac_latch << 2) | 0x03;

	// Compute the full-scale (max) output current.
	const double i_fs = I_FS_NOMINAL + i_bias(m_dac_gain, RES_K(100), RES_K(100), 0);  // R333, R329
	// Adding i_bias() is a guess. There were no details found on the
	// quantitative effects of DAC Gain. But the implementation is qualitatively
	// correct: according to graphs on the datasheet, gain affects i_out
	// proportionally with scale. Furthermore, the adjustment range that results
	// from adding i_bias() (+/- ~0.78V) seems reasonable.

	// The DAC sinks its max current (i_fs) when the input is 0, and sinks no
	// current when the input is 0xffff.
	const double i_out = i_fs * (0xffff - dac_input) / double(0xffff);

	// The current is converted to a voltage by U347 (LF356 op-amp) and
	// surrounding resistors.
	m_vdac = i_out * (DAC_RF + RES_R(332));  // R330
	update_sh();

	// The DAC voltage is scaled down and used as a reference for the ADC. It
	// will be divided by ~2 if ADC Gain is properly calibrated.
	const double adc_gain = RES_K(5) * normalized(m_adc_gain);  // R334
	m_adc_vref = m_vdac * RES_VOLTAGE_DIVIDER(adc_gain + RES_K(18.2), RES_K(20.0));  // R335, R336
}

void prophet5_state::update_vmux()
{
	// The Vmux signal is one of the inputs to the ADC comparator. The other
	// one is the ADC reference (m_adc_vref).

	double vmux_sum = 0;
	int n_active_switches = 0;

	constexpr double POT_V_MAX = 5.0;
	for (int mux = 0; mux < 3; ++mux)
	{
		if (!BIT(m_pot_mux_inh, mux))  // Active low.
		{
			const int pot_index = 8 * mux + m_pot_mux_abc;
			vmux_sum += POT_V_MAX * normalized(m_pots[pot_index]);
			++n_active_switches;
		}
	}

	if (m_seq_cv_enabled)  // U371C (CD4016) control input.
	{
		const double cv_in = MAX_CV_IN * normalized(m_seq_cv_in);

		// The external CV is buffered, scaled, and offsetted by U374A (LM348
		// op-amp) and surrounding circuit.

		// Scale resistor network.
		constexpr double R396 = RES_R(470);
		constexpr double R392 = RES_K(24.9);
		constexpr double R391 = RES_K(30.1);
		const double R386 = RES_K(10) * normalized(m_seq_scale);
		const double scaled_cv = cv_in * RES_VOLTAGE_DIVIDER(R396 + R392 + R386, R391);

		// Offset resistor network.
		constexpr double R389 = RES_K(1);
		constexpr double R390 = RES_M(1);
		constexpr double R385_MAX = RES_K(100);
		const double offset = R389 * i_bias(m_seq_offset, R385_MAX, R390, scaled_cv);

		vmux_sum += scaled_cv + offset;
		++n_active_switches;
	}

	// Each MUX output and the SEQ CV switch output have a 1K resistor to protect
	// from short circuits during startup. Under normal operation, only one
	// switch should be selected. If (buggy) firmware activates more than one,
	// the protection resistors will average out the voltages.

	if (n_active_switches > 0)
		m_adc_vmux = vmux_sum / n_active_switches;
	// Else, Vmux is floating. Happens transiently under normal operation.
	// Leaving m_adc_vmux unchanged when that happens.
}

void prophet5_state::mux_abc_w(u8 data)  // U338, CD4174 latch.
{
	// D0-D2: ABC inputs for all S&H and Tune MUXes.
	m_mux_abc = BIT(data, 0, 3);
	update_sh();

	// D3: -TUNE.
	m_tuning = BIT(data, 3) ? 0 : 1;

	// D4-D5: INH inputs of individual Tune MUXes.
	m_tune_mux_select = ~BIT(data, 4, 2) & 0x03;
}

void prophet5_state::sh_mux_inh_w(u8 data)  // U339, CD4174 latch.
{
	// D0-D4: INH inputs of individual S&H MUXes.
	m_sh_mux_inh = BIT(data, 0, 5);
	update_sh();

	// D5: EN SEQ CV
	m_seq_cv_enabled = BIT(data, 5);
	update_vmux();
}

void prophet5_state::pot_mux_w(u8 data)  // U211, CD4174 latch.
{
	m_pot_mux_abc = BIT(data, 0, 3);
	m_pot_mux_inh = BIT(data, 3, 3);
	update_vmux();
}

void prophet5_state::dac_w(offs_t offset, u8 data)
{
	if (offset == 0)  // Latch low 7 bits.
		m_dac_latch = (m_dac_latch & 0x3f80) | (data & 0x7f);
	else if (offset == 1)  // Latch high 7 bits.
		m_dac_latch = (u16(data & 0x7f) << 7) | (m_dac_latch & 0x007f);
	else
		assert(false);  // Should not happen.
	update_vdac();
}

void prophet5_state::update_gate5()
{
	// U321B (74LS02) -> U331F (CD4049)
	m_audio->voice(4)->gate_w((m_latch_gate5 || m_ext_gate5) ? 1 : 0);
}

void prophet5_state::latch_gate5_w(int state)
{
	m_latch_gate5 = bool(state);
	update_gate5();
}

void prophet5_state::clr_int_w(u8 data)
{
	// Flipflop U330A (4013) R input asserted, making /Q (-> /INT) = 1.
	m_maincpu->set_input_line(INPUT_LINE_IRQ0, CLEAR_LINE);

	// Flipflop U330B (4013) S input asserted, making Q (-> m_ext_gate5) = 1.
	m_ext_gate5 = true;
	update_gate5();

	if (!BIT(m_gate_in->read(), 0))
	{
		// In this case, U330B input R is also asserted. While both R and S are
		// asserted, Q will be 1. But S is only asserted for the duration of the
		// I/O strobe. Once that ends, only R will be asserted, and Q will
		// transition to 0.
		// This situation should not occur, unless the firmware is misbehaving.
		m_ext_gate5 = false;
		update_gate5();
	}
}

TIMER_DEVICE_CALLBACK_MEMBER(prophet5_state::gate_in_delay_elapsed)
{
	// Flipflop U330A (4013) is clocked with D = 1, making /Q (-> /INT) = 0.
	m_maincpu->set_input_line(INPUT_LINE_IRQ0, ASSERT_LINE);
}

void prophet5_state::update_nvram_record()
{
	m_nvram_write_view.select(BIT(m_record->read(), 0));
}

void prophet5_state::memory_map(address_map &map)
{
	map.global_mask(0x9fff);  // Z80 A13 and A14 are not connected.

	// Memory decoding done by U318 (74LS138).
	map(0x0000, 0x0bff).mirror(0x8000).rom();
	map(0x0c00, 0x0fff).readonly().share("nvram");  // 8 x 1K x 1bit RAMs (6508, U301-U308).
	map(0x1000, 0x13ff).mirror(0x8000).ram();  // 2 x 1K x 4bit RAMs (2114, U316, U317).
	map(0x1800, 0x1803).mirror(0x83fc).rw("tune_pit", FUNC(pit8253_device::read), FUNC(pit8253_device::write));
	map(0x8c00, 0x8fff).view(m_nvram_write_view);

	// The "record" switch write protects the NVRAM. When "record" is enabled,
	// U320C (74LS00) will assert RAM /WR when A15=1. When "record" is disabled,
	// /WR will not be asserted, which will result in an NVRAM read.
	m_nvram_write_view[0](0x8c00, 0x8fff).readonly().share("nvram");
	m_nvram_write_view[1](0x8c00, 0x8fff).writeonly().share("nvram");
}

void prophet5_state::io_map(address_map &map)
{
	// The signal names in the comments below (e.g. "CSI0, KBD/SW") match those
	// in the schematics.
	map.global_mask(0x3f);  // Only A0-A5 are used for port decoding.

	// Input port decoding done by U310 A,B,C (74LS00).
	map(0x01, 0x01).mirror(0x3e).r(FUNC(prophet5_state::switch_r));  // CSI0, KBD/SW
	map(0x02, 0x02).mirror(0x3d).r(FUNC(prophet5_state::misc_r));  // CSI1, CASS/MISC
	map(0x04, 0x04).mirror(0x3b).r(FUNC(prophet5_state::adc_r));  // CSI2, ADC

	// Output port decoding for TTL chips done by U319 (74LS138).
	map(0x00, 0x00).mirror(0x07).w(FUNC(prophet5_state::led_drive_w));  // CSOL0, LED DRVR
	map(0x08, 0x08).mirror(0x07).w(FUNC(prophet5_state::led_sink_w));  // CSOL1, LED SINK
	map(0x10, 0x10).mirror(0x07).w(FUNC(prophet5_state::switch_w));  // CSOL2, KBD/SW DRVR
	map(0x18, 0x18).mirror(0x07).w(FUNC(prophet5_state::pot_mux_w));  // CSOL3, POT MUX ADR
	map(0x20, 0x20).mirror(0x07).w("misc_latch", FUNC(output_latch_device::write));  // CSOL4, CASS/TUNE
	map(0x28, 0x28).mirror(0x07).w(FUNC(prophet5_state::clr_int_w));  // CSOL5, CLEAR INT

	// Output port decoding for 15V CMOS chips done by U329 (CD4556).
	map(0x30, 0x30).mirror(0x04).w("program_latch_0", FUNC(output_latch_device::write));  // CSOH0, PROG SW 0
	map(0x31, 0x31).mirror(0x04).w("program_latch_1", FUNC(output_latch_device::write));  // CSOH1, PROG SW 1
	map(0x32, 0x32).mirror(0x04).w("program_latch_2", FUNC(output_latch_device::write));  // CSOH2, PROG SW 2
	map(0x33, 0x33).mirror(0x04).w(FUNC(prophet5_state::mux_abc_w));  // CSOH3, S/H ABC/TUNE
	map(0x38, 0x38).mirror(0x04).w(FUNC(prophet5_state::sh_mux_inh_w));  // CSOH4, S/H
	map(0x39, 0x39).mirror(0x04).w("gate_latch", FUNC(output_latch_device::write));  // CSOH5, GATES
	map(0x3a, 0x3b).mirror(0x04).w(FUNC(prophet5_state::dac_w));  // CSOH6, DAC LSB - CSOH7, DAC MSB
}

void prophet5_state::machine_start()
{
	save_item(NAME(m_switch_row));
	save_item(NAME(m_mux_abc));
	save_item(NAME(m_sh_mux_inh));
	save_item(NAME(m_seq_cv_enabled));
	save_item(NAME(m_dac_latch));
	save_item(NAME(m_pot_mux_abc));
	save_item(NAME(m_pot_mux_inh));
	save_item(NAME(m_vdac));
	save_item(NAME(m_adc_vmux));
	save_item(NAME(m_adc_vref));
	save_item(NAME(m_tune_counter_out));
	save_item(NAME(m_latch_gate5));
	save_item(NAME(m_ext_gate5));

	m_tune_mux_select.resolve();
	m_tuning.resolve();
	for (auto &led_row : m_leds)
		for (auto &led : led_row)
			led.resolve();
}

void prophet5_state::machine_reset()
{
	update_nvram_record();
}

void prophet5_state::prophet5rev30(machine_config &config)
{
	Z80(config, m_maincpu, 5_MHz_XTAL / 2);  // Divided by U325.
	m_maincpu->set_addrmap(AS_PROGRAM, &prophet5_state::memory_map);
	m_maincpu->set_addrmap(AS_IO, &prophet5_state::io_map);

	NVRAM(config, "nvram", nvram_device::DEFAULT_ALL_0);

	auto &pit = PIT8253(config, "tune_pit");  // U315
	pit.out_handler<0>().set("tune_pit", FUNC(pit8253_device::write_gate2)).invert();
	pit.out_handler<0>().append([this] (int state) { m_tune_counter_out = bool(state); });
	pit.out_handler<1>().set(m_audio, FUNC(prophet5_audio_device::a440_w));
	pit.set_clk<1>(5_MHz_XTAL / 2);
	pit.set_clk<2>(5_MHz_XTAL / 2);

	TTL7474(config, m_tune_ff, 0).comp_output_cb().set("tune_pit", FUNC(pit8253_device::write_clk0));

	TIMER(config, m_gate_in_delay).configure_generic(FUNC(prophet5_state::gate_in_delay_elapsed));

	PWM_DISPLAY(config, m_led_matrix_pwm).set_size(8, 5);
	m_led_matrix_pwm->output_x().set(FUNC(prophet5_state::led_update_w));

	PWM_DISPLAY(config, m_digit_pwm).set_size(2, 7);
	m_digit_pwm->set_segmask(0x03, 0x7f);

	config.set_default_layout(layout_sequential_prophet5);

	PROPHET5_AUDIO(config, m_audio);

	auto &u332 = OUTPUT_LATCH(config, "misc_latch");
	u332.bit_handler<0>().set(m_tune_ff, FUNC(ttl7474_device::clear_w));
	u332.bit_handler<1>().set(m_tune_ff, FUNC(ttl7474_device::preset_w));
	u332.bit_handler<2>().set_output("cassette_out");
	u332.bit_handler<3>().set(m_tune_ff, FUNC(ttl7474_device::d_w));
	u332.bit_handler<4>().set("tune_pit", FUNC(pit8253_device::write_gate0));
	u332.bit_handler<5>().set("tune_pit", FUNC(pit8253_device::write_gate1));
	u332.bit_handler<5>().append(m_audio, FUNC(prophet5_audio_device::select_a440_w));

	auto &u335 = OUTPUT_LATCH(config, "program_latch_0");
	u335.bit_handler<0>().set_output("osc_a_sqr");
	u335.bit_handler<1>().set_output("osc_a_saw");
	u335.bit_handler<2>().set_output("osc_a_sync");
	u335.bit_handler<3>().set_output("osc_b_saw");
	u335.bit_handler<4>().set_output("osc_b_tri");
	u335.bit_handler<5>().set_output("osc_b_sqr");
	u335.bit_handler<6>().set_output("osc_b_kbd");  // Actually U341A (4013 flipflop).

	auto &u334 = OUTPUT_LATCH(config, "program_latch_1");
	u334.bit_handler<0>().set_output("pmod_freq_a");
	u334.bit_handler<1>().set_output("pmod_pw_a");
	u334.bit_handler<2>().set_output("pmod_filt");
	u334.bit_handler<3>().set_output("lfo_saw");
	u334.bit_handler<4>().set_output("lfo_tri");
	u334.bit_handler<5>().set_output("lfo_sqr");
	u334.bit_handler<6>().set(m_audio, FUNC(prophet5_audio_device::filt_kbd_s_w));  // Actually U341B (4013 flipflop).

	auto &u333 = OUTPUT_LATCH(config, "program_latch_2");
	u333.bit_handler<0>().set_output("wmod_freq_a");
	u333.bit_handler<1>().set_output("wmod_freq_b");
	u333.bit_handler<2>().set_output("wmod_pw_a");
	u333.bit_handler<3>().set_output("wmod_pw_b");
	u333.bit_handler<4>().set(m_audio, FUNC(prophet5_audio_device::wmod_filt_s_w));
	u333.bit_handler<5>().set_output("osc_b_lo");

	auto &u340 = OUTPUT_LATCH(config, "gate_latch");
	u340.bit_handler<0>().set([this] (int state) { m_audio->voice(0)->gate_w(state); });
	u340.bit_handler<1>().set([this] (int state) { m_audio->voice(1)->gate_w(state); });
	u340.bit_handler<2>().set([this] (int state) { m_audio->voice(2)->gate_w(state); });
	u340.bit_handler<3>().set([this] (int state) { m_audio->voice(3)->gate_w(state); });
	u340.bit_handler<4>().set(FUNC(prophet5_state::latch_gate5_w));
	u340.bit_handler<5>().set_output("gate_out");  // J5, SEQ TRIG OUT
}

DECLARE_INPUT_CHANGED_MEMBER(prophet5_state::record_changed)
{
	update_nvram_record();
}

DECLARE_INPUT_CHANGED_MEMBER(prophet5_state::gate_in_changed)
{
	if (newval)
	{
		// An RC circuit adds a delay between receiving the gate-in signal and
		// asserting /INT. `DT` is the time it takes for the RC network to
		// discharge from 5V to 2.5V and trip the inverter (U331A, CD4049). This
		// is ~1.4ms nominal. The schematic says "2ms delay". The actual delay
		// is not well-specified, since it depends on the trip point of the
		// inverter.
		const double DT = -RES_K(100) * CAP_U(0.02) * log(2.5 / 5.0);  // R311, C316
		m_gate_in_delay->adjust(attotime::from_double(DT));
	}
	else
	{
		m_gate_in_delay->reset();
		m_ext_gate5 = false;  // Flipflop U330B (4013) R input asserted, making Q (-> m_ext_gate5) = 0.
		update_gate5();
	}
}

DECLARE_INPUT_CHANGED_MEMBER(prophet5_state::pot_adjusted)
{
	update_vmux();
}

DECLARE_INPUT_CHANGED_MEMBER(prophet5_state::dac_trimmer_adjusted)
{
	update_vdac();
}

DECLARE_INPUT_CHANGED_MEMBER(prophet5_state::adc_trimmer_adjusted)
{
	update_vdac();
}

DECLARE_INPUT_CHANGED_MEMBER(prophet5_state::cv_in_changed)
{
	update_vmux();
}

DECLARE_INPUT_CHANGED_MEMBER(prophet5_state::seq_trimmer_adjusted)
{
	update_vmux();
}

INPUT_PORTS_START(prophet5)
	PORT_START("switch_row_0")
	PORT_BIT(0x01, IP_ACTIVE_HIGH, IPT_OTHER) PORT_NAME("OSC A SQR")
	PORT_BIT(0x02, IP_ACTIVE_HIGH, IPT_OTHER) PORT_NAME("OSC A SAW")
	PORT_BIT(0x04, IP_ACTIVE_HIGH, IPT_OTHER) PORT_NAME("OSC A SYNC")
	PORT_BIT(0x08, IP_ACTIVE_HIGH, IPT_OTHER) PORT_NAME("OSC B SAW")
	PORT_BIT(0x10, IP_ACTIVE_HIGH, IPT_OTHER) PORT_NAME("OSC B TRI")
	PORT_BIT(0x20, IP_ACTIVE_HIGH, IPT_OTHER) PORT_NAME("OSC B SQR")
	PORT_BIT(0x40, IP_ACTIVE_HIGH, IPT_OTHER) PORT_NAME("OSC B KBD")
	PORT_BIT(0x80, IP_ACTIVE_HIGH, IPT_OTHER) PORT_NAME("UNISON") PORT_CODE(KEYCODE_U)

	PORT_START("switch_row_1")
	PORT_BIT(0x01, IP_ACTIVE_HIGH, IPT_OTHER) PORT_NAME("PMOD FREQ A")
	PORT_BIT(0x02, IP_ACTIVE_HIGH, IPT_OTHER) PORT_NAME("PMOD PW A")
	PORT_BIT(0x04, IP_ACTIVE_HIGH, IPT_OTHER) PORT_NAME("PMOD FILT")
	PORT_BIT(0x08, IP_ACTIVE_HIGH, IPT_OTHER) PORT_NAME("LFO SAW")
	PORT_BIT(0x10, IP_ACTIVE_HIGH, IPT_OTHER) PORT_NAME("LFO TRI")
	PORT_BIT(0x20, IP_ACTIVE_HIGH, IPT_OTHER) PORT_NAME("LFO SQR")
	PORT_BIT(0x40, IP_ACTIVE_HIGH, IPT_OTHER) PORT_NAME("FILT KBD") PORT_CODE(KEYCODE_K)
	PORT_BIT(0x80, IP_ACTIVE_HIGH, IPT_OTHER) PORT_NAME("RELEASE")

	PORT_START("switch_row_2")
	PORT_BIT(0x01, IP_ACTIVE_HIGH, IPT_OTHER) PORT_NAME("WMOD FREQ A")
	PORT_BIT(0x02, IP_ACTIVE_HIGH, IPT_OTHER) PORT_NAME("WMOD FREQ B")
	PORT_BIT(0x04, IP_ACTIVE_HIGH, IPT_OTHER) PORT_NAME("WMOD PW A")
	PORT_BIT(0x08, IP_ACTIVE_HIGH, IPT_OTHER) PORT_NAME("WMOD PW B")
	PORT_BIT(0x10, IP_ACTIVE_HIGH, IPT_OTHER) PORT_NAME("WMOD FILT")
	PORT_BIT(0x20, IP_ACTIVE_HIGH, IPT_OTHER) PORT_NAME("OSC B LO")
	PORT_BIT(0x40, IP_ACTIVE_HIGH, IPT_UNUSED)
	PORT_BIT(0x80, IP_ACTIVE_HIGH, IPT_UNUSED)

	PORT_START("switch_row_3")
	PORT_BIT(0x01, IP_ACTIVE_HIGH, IPT_OTHER) PORT_NAME("PS1") PORT_CODE(KEYCODE_1)
	PORT_BIT(0x02, IP_ACTIVE_HIGH, IPT_OTHER) PORT_NAME("PS2") PORT_CODE(KEYCODE_2)
	PORT_BIT(0x04, IP_ACTIVE_HIGH, IPT_OTHER) PORT_NAME("PS3") PORT_CODE(KEYCODE_3)
	PORT_BIT(0x08, IP_ACTIVE_HIGH, IPT_OTHER) PORT_NAME("PS4") PORT_CODE(KEYCODE_4)
	PORT_BIT(0x10, IP_ACTIVE_HIGH, IPT_OTHER) PORT_NAME("PS5") PORT_CODE(KEYCODE_5)
	PORT_BIT(0x20, IP_ACTIVE_HIGH, IPT_OTHER) PORT_NAME("PS6") PORT_CODE(KEYCODE_6)
	PORT_BIT(0x40, IP_ACTIVE_HIGH, IPT_OTHER) PORT_NAME("PS7") PORT_CODE(KEYCODE_7)
	PORT_BIT(0x80, IP_ACTIVE_HIGH, IPT_OTHER) PORT_NAME("PS8") PORT_CODE(KEYCODE_8)

	PORT_START("switch_row_4")
	PORT_BIT(0x01, IP_ACTIVE_HIGH, IPT_OTHER) PORT_NAME("RECORD") PORT_CODE(KEYCODE_R)
	PORT_BIT(0x02, IP_ACTIVE_HIGH, IPT_OTHER) PORT_NAME("BANK SELECT") PORT_CODE(KEYCODE_S)
	PORT_BIT(0x04, IP_ACTIVE_HIGH, IPT_OTHER) PORT_NAME("A-440") PORT_CODE(KEYCODE_A)
	PORT_BIT(0x08, IP_ACTIVE_HIGH, IPT_OTHER) PORT_NAME("TUNE")
	PORT_BIT(0x10, IP_ACTIVE_HIGH, IPT_OTHER) PORT_NAME("TO CASS")
	PORT_BIT(0x20, IP_ACTIVE_HIGH, IPT_OTHER) PORT_NAME("FROM CASS")
	PORT_BIT(0x40, IP_ACTIVE_HIGH, IPT_UNUSED)
	PORT_BIT(0x80, IP_ACTIVE_HIGH, IPT_OTHER) PORT_NAME("PRESET") PORT_CODE(KEYCODE_P)

	PORT_START("switch_row_5")
	PORT_BIT(0xff, IP_ACTIVE_HIGH, IPT_UNUSED)

	PORT_START("switch_row_6")
	PORT_BIT(0xff, IP_ACTIVE_HIGH, IPT_UNUSED)

	PORT_START("switch_row_7")
	PORT_BIT(0xff, IP_ACTIVE_HIGH, IPT_UNUSED)

	PORT_START("switch_row_8")  // C0 - G0 in schematic.
	PORT_BIT(0x01, IP_ACTIVE_HIGH, IPT_OTHER) PORT_GM_C2
	PORT_BIT(0x02, IP_ACTIVE_HIGH, IPT_OTHER) PORT_GM_CS2
	PORT_BIT(0x04, IP_ACTIVE_HIGH, IPT_OTHER) PORT_GM_D2
	PORT_BIT(0x08, IP_ACTIVE_HIGH, IPT_OTHER) PORT_GM_DS2
	PORT_BIT(0x10, IP_ACTIVE_HIGH, IPT_OTHER) PORT_GM_E2
	PORT_BIT(0x20, IP_ACTIVE_HIGH, IPT_OTHER) PORT_GM_F2
	PORT_BIT(0x40, IP_ACTIVE_HIGH, IPT_OTHER) PORT_GM_FS2
	PORT_BIT(0x80, IP_ACTIVE_HIGH, IPT_OTHER) PORT_GM_G2

	PORT_START("switch_row_9")  // G#0 - D#1
	PORT_BIT(0x01, IP_ACTIVE_HIGH, IPT_OTHER) PORT_GM_GS2
	PORT_BIT(0x02, IP_ACTIVE_HIGH, IPT_OTHER) PORT_GM_A2 PORT_CODE(KEYCODE_Z)
	PORT_BIT(0x04, IP_ACTIVE_HIGH, IPT_OTHER) PORT_GM_AS2
	PORT_BIT(0x08, IP_ACTIVE_HIGH, IPT_OTHER) PORT_GM_B2
	PORT_BIT(0x10, IP_ACTIVE_HIGH, IPT_OTHER) PORT_GM_C3
	PORT_BIT(0x20, IP_ACTIVE_HIGH, IPT_OTHER) PORT_GM_CS3
	PORT_BIT(0x40, IP_ACTIVE_HIGH, IPT_OTHER) PORT_GM_D3
	PORT_BIT(0x80, IP_ACTIVE_HIGH, IPT_OTHER) PORT_GM_DS3

	PORT_START("switch_row_10")  // E1 - B1
	PORT_BIT(0x01, IP_ACTIVE_HIGH, IPT_OTHER) PORT_GM_E3
	PORT_BIT(0x02, IP_ACTIVE_HIGH, IPT_OTHER) PORT_GM_F3
	PORT_BIT(0x04, IP_ACTIVE_HIGH, IPT_OTHER) PORT_GM_FS3
	PORT_BIT(0x08, IP_ACTIVE_HIGH, IPT_OTHER) PORT_GM_G3
	PORT_BIT(0x10, IP_ACTIVE_HIGH, IPT_OTHER) PORT_GM_GS3
	PORT_BIT(0x20, IP_ACTIVE_HIGH, IPT_OTHER) PORT_GM_A3 PORT_CODE(KEYCODE_X)
	PORT_BIT(0x40, IP_ACTIVE_HIGH, IPT_OTHER) PORT_GM_AS3
	PORT_BIT(0x80, IP_ACTIVE_HIGH, IPT_OTHER) PORT_GM_B3

	PORT_START("switch_row_11")  // C2 - G2
	PORT_BIT(0x01, IP_ACTIVE_HIGH, IPT_OTHER) PORT_GM_C4
	PORT_BIT(0x02, IP_ACTIVE_HIGH, IPT_OTHER) PORT_GM_CS4
	PORT_BIT(0x04, IP_ACTIVE_HIGH, IPT_OTHER) PORT_GM_D4
	PORT_BIT(0x08, IP_ACTIVE_HIGH, IPT_OTHER) PORT_GM_DS4
	PORT_BIT(0x10, IP_ACTIVE_HIGH, IPT_OTHER) PORT_GM_E4
	PORT_BIT(0x20, IP_ACTIVE_HIGH, IPT_OTHER) PORT_GM_F4
	PORT_BIT(0x40, IP_ACTIVE_HIGH, IPT_OTHER) PORT_GM_FS4
	PORT_BIT(0x80, IP_ACTIVE_HIGH, IPT_OTHER) PORT_GM_G4

	PORT_START("switch_row_12")  // G#2 - D#3
	PORT_BIT(0x01, IP_ACTIVE_HIGH, IPT_OTHER) PORT_GM_GS4
	PORT_BIT(0x02, IP_ACTIVE_HIGH, IPT_OTHER) PORT_GM_A4 PORT_CODE(KEYCODE_C)
	PORT_BIT(0x04, IP_ACTIVE_HIGH, IPT_OTHER) PORT_GM_AS4
	PORT_BIT(0x08, IP_ACTIVE_HIGH, IPT_OTHER) PORT_GM_B4
	PORT_BIT(0x10, IP_ACTIVE_HIGH, IPT_OTHER) PORT_GM_C5
	PORT_BIT(0x20, IP_ACTIVE_HIGH, IPT_OTHER) PORT_GM_CS5
	PORT_BIT(0x40, IP_ACTIVE_HIGH, IPT_OTHER) PORT_GM_D4
	PORT_BIT(0x80, IP_ACTIVE_HIGH, IPT_OTHER) PORT_GM_DS4

	PORT_START("switch_row_13")  // E3 - B3
	PORT_BIT(0x01, IP_ACTIVE_HIGH, IPT_OTHER) PORT_GM_E5
	PORT_BIT(0x02, IP_ACTIVE_HIGH, IPT_OTHER) PORT_GM_F5
	PORT_BIT(0x04, IP_ACTIVE_HIGH, IPT_OTHER) PORT_GM_FS4
	PORT_BIT(0x08, IP_ACTIVE_HIGH, IPT_OTHER) PORT_GM_G5
	PORT_BIT(0x10, IP_ACTIVE_HIGH, IPT_OTHER) PORT_GM_GS5
	PORT_BIT(0x20, IP_ACTIVE_HIGH, IPT_OTHER) PORT_GM_A5 PORT_CODE(KEYCODE_V)
	PORT_BIT(0x40, IP_ACTIVE_HIGH, IPT_OTHER) PORT_GM_AS5
	PORT_BIT(0x80, IP_ACTIVE_HIGH, IPT_OTHER) PORT_GM_B5

	PORT_START("switch_row_14")  // C4 - G4
	PORT_BIT(0x01, IP_ACTIVE_HIGH, IPT_OTHER) PORT_GM_C6
	PORT_BIT(0x02, IP_ACTIVE_HIGH, IPT_OTHER) PORT_GM_CS6
	PORT_BIT(0x04, IP_ACTIVE_HIGH, IPT_OTHER) PORT_GM_D6
	PORT_BIT(0x08, IP_ACTIVE_HIGH, IPT_OTHER) PORT_GM_DS6
	PORT_BIT(0x10, IP_ACTIVE_HIGH, IPT_OTHER) PORT_GM_E6
	PORT_BIT(0x20, IP_ACTIVE_HIGH, IPT_OTHER) PORT_GM_F6
	PORT_BIT(0x40, IP_ACTIVE_HIGH, IPT_OTHER) PORT_GM_FS6
	PORT_BIT(0x80, IP_ACTIVE_HIGH, IPT_OTHER) PORT_GM_G6

	PORT_START("switch_row_15")  // G#4 - C5
	PORT_BIT(0x01, IP_ACTIVE_HIGH, IPT_OTHER) PORT_GM_GS6
	PORT_BIT(0x02, IP_ACTIVE_HIGH, IPT_OTHER) PORT_GM_A6 PORT_CODE(KEYCODE_B)
	PORT_BIT(0x04, IP_ACTIVE_HIGH, IPT_OTHER) PORT_GM_AS6
	PORT_BIT(0x08, IP_ACTIVE_HIGH, IPT_OTHER) PORT_GM_B6
	PORT_BIT(0x10, IP_ACTIVE_HIGH, IPT_OTHER) PORT_GM_C7
	PORT_BIT(0x20, IP_ACTIVE_HIGH, IPT_UNUSED)
	PORT_BIT(0x40, IP_ACTIVE_HIGH, IPT_UNUSED)
	PORT_BIT(0x80, IP_ACTIVE_HIGH, IPT_UNUSED)

	// NVRAM (patch memory) write protect switch on back panel. It electrically
	// disables writes to the NVRAM (blocks the /WR signal, look for
	// m_nvram_write_view), and its state can be read by the firmware (see
	// misc_r()).
	PORT_START("record")
	PORT_CONFNAME(0x01, 0x01, "RECORD EN DIS")
		PORT_CHANGED_MEMBER(DEVICE_SELF, FUNC(prophet5_state::record_changed), 0)
	PORT_CONFSETTING(0x00, "Disable")
	PORT_CONFSETTING(0x01, "Enable")

	PORT_START("gate_in_connected")
	PORT_CONFNAME(0x01, 0x00, "EXT GATE EN")
	PORT_CONFSETTING(0x00, "Not connected")
	PORT_CONFSETTING(0x01, "Connected")

	PORT_START("gate_in")
	PORT_BIT(0x01, IP_ACTIVE_HIGH, IPT_OTHER) PORT_NAME("SEQ GATE IN") PORT_CODE(KEYCODE_G)
		PORT_CHANGED_MEMBER(DEVICE_SELF, FUNC(prophet5_state::gate_in_changed), 0)

	PORT_START("release_footswitch")
	PORT_BIT(0x01, IP_ACTIVE_LOW, IPT_OTHER) PORT_NAME("REL FT SW")

	PORT_START("test_points")
	// According to the schematic, TP301 and TP304 have pull-down resistors, and
	// TP306 does not have a resistor.
	PORT_BIT(0x02, IP_ACTIVE_HIGH, IPT_OTHER) PORT_NAME("TP301") PORT_CODE(KEYCODE_T)
	PORT_BIT(0x10, IP_ACTIVE_HIGH, IPT_OTHER) PORT_NAME("TP304")
	PORT_BIT(0x40, IP_ACTIVE_HIGH, IPT_OTHER) PORT_NAME("TP306")

	// All knob potentiometers are 10K linear, unless otherwise noted.

	PORT_START("pot_0")  // R217
	PORT_ADJUSTER(0, "GLIDE") PORT_MINMAX(0, 255)
		PORT_CHANGED_MEMBER(DEVICE_SELF, FUNC(prophet5_state::pot_adjusted), 0)

	PORT_START("pot_1")  // R211
	PORT_ADJUSTER(127, "LFO FREQ") PORT_MINMAX(0, 255)
		PORT_CHANGED_MEMBER(DEVICE_SELF, FUNC(prophet5_state::pot_adjusted), 1)

	PORT_START("pot_2")  // R216
	PORT_ADJUSTER(0, "WMOD SRC MIX") PORT_MINMAX(0, 255)
		PORT_CHANGED_MEMBER(DEVICE_SELF, FUNC(prophet5_state::pot_adjusted), 2)

	PORT_START("pot_3")  // R202
	PORT_ADJUSTER(0, "PMOD OSC B") PORT_MINMAX(0, 255)
		PORT_CHANGED_MEMBER(DEVICE_SELF, FUNC(prophet5_state::pot_adjusted), 3)

	PORT_START("pot_4")  // R201
	PORT_ADJUSTER(0, "PMOD FILT ENV") PORT_MINMAX(0, 255)
		PORT_CHANGED_MEMBER(DEVICE_SELF, FUNC(prophet5_state::pot_adjusted), 4)

	PORT_START("pot_5")  // R204
	PORT_ADJUSTER(127, "OSC A FREQ") PORT_MINMAX(0, 255)
		PORT_CHANGED_MEMBER(DEVICE_SELF, FUNC(prophet5_state::pot_adjusted), 5)

	PORT_START("pot_6")  // R213
	PORT_ADJUSTER(127, "OSC B FREQ") PORT_MINMAX(0, 255)
		PORT_CHANGED_MEMBER(DEVICE_SELF, FUNC(prophet5_state::pot_adjusted), 6)

	PORT_START("pot_7")  // R214
	PORT_ADJUSTER(127, "OSC B FINE") PORT_MINMAX(0, 255)
		PORT_CHANGED_MEMBER(DEVICE_SELF, FUNC(prophet5_state::pot_adjusted), 7)

	PORT_START("pot_8")  // R101
	PORT_ADJUSTER(255, "FILT CUTOFF") PORT_MINMAX(0, 255)
		PORT_CHANGED_MEMBER(DEVICE_SELF, FUNC(prophet5_state::pot_adjusted), 8)

	PORT_START("pot_9")  // R103
	PORT_ADJUSTER(0, "FILT ENV AMT") PORT_MINMAX(0, 255)
		PORT_CHANGED_MEMBER(DEVICE_SELF, FUNC(prophet5_state::pot_adjusted), 9)

	PORT_START("pot_10")  // R208
	PORT_ADJUSTER(255, "MIX OSC B") PORT_MINMAX(0, 255)
		PORT_CHANGED_MEMBER(DEVICE_SELF, FUNC(prophet5_state::pot_adjusted), 10)

	PORT_START("pot_11")  // R215
	PORT_ADJUSTER(127, "OSC B PW") PORT_MINMAX(0, 255)
		PORT_CHANGED_MEMBER(DEVICE_SELF, FUNC(prophet5_state::pot_adjusted), 11)

	PORT_START("pot_12")  // R207
	PORT_ADJUSTER(255, "MIX OSC A") PORT_MINMAX(0, 255)
		PORT_CHANGED_MEMBER(DEVICE_SELF, FUNC(prophet5_state::pot_adjusted), 12)

	PORT_START("pot_13")  // R205
	PORT_ADJUSTER(127, "OSC A PW") PORT_MINMAX(0, 255)
		PORT_CHANGED_MEMBER(DEVICE_SELF, FUNC(prophet5_state::pot_adjusted), 13)

	PORT_START("pot_14")  // R210
	PORT_ADJUSTER(0, "MIX NOISE") PORT_MINMAX(0, 255)
		PORT_CHANGED_MEMBER(DEVICE_SELF, FUNC(prophet5_state::pot_adjusted), 14)

	PORT_START("pot_15")  // R102
	PORT_ADJUSTER(0, "FILT RESONANCE") PORT_MINMAX(0, 255)
		PORT_CHANGED_MEMBER(DEVICE_SELF, FUNC(prophet5_state::pot_adjusted), 15)

	PORT_START("pot_16")  // R105
	PORT_ADJUSTER(10, "FILT ATTACK") PORT_MINMAX(0, 255)
		PORT_CHANGED_MEMBER(DEVICE_SELF, FUNC(prophet5_state::pot_adjusted), 16)

	PORT_START("pot_17")  // R106
	PORT_ADJUSTER(10, "FILT DECAY") PORT_MINMAX(0, 255)
		PORT_CHANGED_MEMBER(DEVICE_SELF, FUNC(prophet5_state::pot_adjusted), 17)

	PORT_START("pot_18")  // R107
	PORT_ADJUSTER(255, "FILT SUSTAIN") PORT_MINMAX(0, 255)
		PORT_CHANGED_MEMBER(DEVICE_SELF, FUNC(prophet5_state::pot_adjusted), 18)

	PORT_START("pot_19")  // R108
	PORT_ADJUSTER(20, "FILT RELEASE") PORT_MINMAX(0, 255)
		PORT_CHANGED_MEMBER(DEVICE_SELF, FUNC(prophet5_state::pot_adjusted), 19)

	PORT_START("pot_20")  // R109
	PORT_ADJUSTER(10, "AMP ATTACK") PORT_MINMAX(0, 255)
		PORT_CHANGED_MEMBER(DEVICE_SELF, FUNC(prophet5_state::pot_adjusted), 20)

	PORT_START("pot_21")  // R110
	PORT_ADJUSTER(10, "AMP DECAY") PORT_MINMAX(0, 255)
		PORT_CHANGED_MEMBER(DEVICE_SELF, FUNC(prophet5_state::pot_adjusted), 21)

	PORT_START("pot_22")  // R111
	PORT_ADJUSTER(255, "AMP SUSTAIN") PORT_MINMAX(0, 255)
		PORT_CHANGED_MEMBER(DEVICE_SELF, FUNC(prophet5_state::pot_adjusted), 22)

	PORT_START("pot_23")  // R112
	PORT_ADJUSTER(20, "AMP RELEASE") PORT_MINMAX(0, 255)
		PORT_CHANGED_MEMBER(DEVICE_SELF, FUNC(prophet5_state::pot_adjusted), 23)

	PORT_START("pot_tune")  // R104, 100K, linear
	PORT_ADJUSTER(50, "MASTER TUNE")

	PORT_START("pot_volume")  // R113, 100K, linear
	PORT_ADJUSTER(90, "VOLUME")
		PORT_CHANGED_MEMBER(AUDIO_TAG, FUNC(prophet5_audio_device::master_volume_changed), 0)

	PORT_START("wheel_pitch")  // R1, 100K, linear
	PORT_BIT(0xff, 50, IPT_PADDLE) PORT_NAME("PITCH WHEEL") PORT_MINMAX(0, 100)
		PORT_SENSITIVITY(30) PORT_KEYDELTA(15) PORT_CENTERDELTA(30)

	PORT_START("wheel_mod")  // R2, 100K, linear
	PORT_ADJUSTER(0, "MOD WHEEL")
		PORT_CHANGED_MEMBER(AUDIO_TAG, FUNC(prophet5_audio_device::mod_wheel_changed), 0)

	PORT_START("trimmer_dac_gain")  // R333, 100K trimmer.
	// Default value based on calibration instructions, with a small error due
	// to adjuster resolution.
	PORT_ADJUSTER(50, "TRIMMER: DAC GAIN") PORT_CHANGED_MEMBER(DEVICE_SELF, FUNC(prophet5_state::dac_trimmer_adjusted), 0)

	PORT_START("trimmer_adc_gain")  // R334, 5K trimmer.
	// Default value calibrated for the required: Vadcref = Vdac / 2.
	PORT_ADJUSTER(36, "TRIMMER: ADC GAIN") PORT_CHANGED_MEMBER(DEVICE_SELF, FUNC(prophet5_state::adc_trimmer_adjusted), 0)

	PORT_START("cv_in_seq")  // J2, external CV input.
	PORT_ADJUSTER(50, "CV IN: SEQ") PORT_CHANGED_MEMBER(DEVICE_SELF, FUNC(prophet5_state::cv_in_changed), 0)

	PORT_START("trimmer_seq_offset")  // R385, 100K trimmer.
	PORT_ADJUSTER(50, "TRIMMER: SEQ OFFSET") PORT_CHANGED_MEMBER(DEVICE_SELF, FUNC(prophet5_state::seq_trimmer_adjusted), 0)

	PORT_START("trimmer_seq_scale")  // R386, 10K trimmer.
	// Default value calibrated for ADC_CV_SEQ_IN = CV_SEQ_IN / 2, with some
	// error due to adjuster resolution. Exact calibration works out to 47.3.
	PORT_ADJUSTER(47, "TRIMMER: SEQ SCALE") PORT_CHANGED_MEMBER(DEVICE_SELF, FUNC(prophet5_state::seq_trimmer_adjusted), 1)

	PORT_START("cv_in_amp_connected")
	PORT_CONFNAME(0x01, 0x00, "AMPLIFIER CV IN")
		PORT_CHANGED_MEMBER(AUDIO_TAG, FUNC(prophet5_audio_device::master_volume_changed), 0)
	PORT_CONFSETTING(0x00, "Not connected")
	PORT_CONFSETTING(0x01, "Connected")

	PORT_START("cv_in_amp")  // J7, master volume CV input.
	PORT_ADJUSTER(50, "CV IN: AMPLIFIER")
		PORT_CHANGED_MEMBER(AUDIO_TAG, FUNC(prophet5_audio_device::master_volume_changed), 0)

	PORT_START("cv_in_filter")  // J6, filter cutoff CV input.
	// An input of 0V has the same effect as the input not being connected.
	PORT_ADJUSTER(0, "CV IN: FILTER")
		PORT_CHANGED_MEMBER(AUDIO_TAG, FUNC(prophet5_audio_device::filter_cv_in_changed), 0)
INPUT_PORTS_END

ROM_START(prophet5rev30)
	ROM_REGION(0xc00, "maincpu", 0)  // 3 x 2708 1Kbyte ROMS.
	ROM_LOAD("0.v8.1.u312", 0x000000, 0x000400, CRC(6337d2ae) SHA1(bad79f6475dc0a8bb139ea0a12258cb3e5bfa0be))
	ROM_LOAD("1.v8.1.u313", 0x000400, 0x000400, CRC(1e334fd3) SHA1(276b7abf4a13fbae0d09e869f786b3073ee82504))
	ROM_LOAD("2.v8.1.u314", 0x000800, 0x000400, CRC(ffafaa95) SHA1(9d119fb22270d45e34c1f16899453ae7469d7d20))
ROM_END

}  // anonymous namespace

// Prophet 5 Rev 3.0, serial numbers 1301-2285.
SYST(1980, prophet5rev30, 0, 0, prophet5rev30, prophet5, prophet5_state, empty_init, "Sequential Circuits", "Prophet 5 (Model 1000) Rev 3.0", MACHINE_NOT_WORKING | MACHINE_NO_SOUND | MACHINE_SUPPORTS_SAVE)
